import * as net from 'net';
import Debug from 'debug';

const debug = Debug('@signageos/front-display:Synchronization:Socket:TcpClient');

export class TcpClient {
	private debugId: string = Math.random().toString(36).substring(2, 9);
	private host: string;
	private port: number;
	private connectionTimeoutMs?: number;
	private socket: net.Socket;

	constructor({ host, port, connectionTimeoutMs }: { host: string; port: number; connectionTimeoutMs?: number }) {
		this.host = host;
		this.port = port;
		this.connectionTimeoutMs = connectionTimeoutMs;
		this.socket = new net.Socket();
	}

	public connect(): Promise<void> {
		debug(`${this.debugId} connect to ${this.host}:${this.port}`);

		if (this.connectionTimeoutMs) {
			this.socket.setTimeout(this.connectionTimeoutMs);
		}

		return new Promise<void>((resolve, reject) => {
			const cleanupListeners = () => {
				this.socket.removeAllListeners('connect');
				this.socket.removeAllListeners('error');
				this.socket.removeAllListeners('timeout');
			};

			this.socket.once('connect', () => {
				debug(`${this.debugId} successfully connected`);
				cleanupListeners();
				resolve();
			});

			this.socket.once('error', (error) => {
				debug(`${this.debugId} failed to connect`, error);
				cleanupListeners();
				reject(error);
			});

			this.socket.once('timeout', () => {
				debug(`${this.debugId} timeout`);
				cleanupListeners();
				this.socket.destroy();
				reject(new Error('socket timeout'));
			});

			this.socket.connect(this.port, this.host);
		});
	}

	public async close(): Promise<void> {
		debug(`${this.debugId} closing`);

		await new Promise<void>((resolve) => {
			this.socket.end(() => {
				debug(`${this.debugId} closed`);
				resolve();
			});
		});

		this.socket.destroy();
	}

	public send(data: Buffer) {
		return new Promise<void>((resolve, reject) => {
			this.socket.write(data, (error) => {
				if (error) {
					reject(error);
				} else {
					resolve();
				}
			});
		});
	}
}
