import { EventEmitter } from 'events';
import Debug from 'debug';
import { IUdpSocket, UdpSocketEvent } from '../Socket/Udp/IUdpSocket';

const debug = Debug('@signageos/front-display:PeerNetwork:PeerDiscoveryServiceMessageSender');

enum MessageType {
	Query = 'query',
	Announce = 'announce',
	Renounce = 'renounce',
	Ping = 'ping',
}

interface BaseMessage {
	sourceId: string;
	type: MessageType;
}

function isMessage(message: unknown): message is { type: string } {
	return typeof message === 'object' && message !== null && 'type' in message;
}

function isBaseMessage(message: unknown): message is BaseMessage {
	const messageTypes = Object.keys(MessageType).map((key) => MessageType[key as keyof typeof MessageType]);
	return isMessage(message) && messageTypes.indexOf(message.type as MessageType) !== -1;
}

export interface QueryMessage extends BaseMessage {
	type: MessageType.Query;
}

export interface AnnounceMessage extends BaseMessage {
	type: MessageType.Announce;
	address: string;
	port: number;
}

export interface RenounceMessage extends BaseMessage {
	type: MessageType.Renounce;
}

export interface PingMessage extends BaseMessage {
	type: MessageType.Ping;
}

enum Event {
	Query = 'query',
	Announce = 'announce',
	Renounce = 'renounce',
	Ping = 'ping',
	Data = 'data',
}

/**
 * Companion class to PeerDiscoveryService that handles sending messages.
 */
export class PeerDiscoveryServiceMessageSender {
	private emitter: EventEmitter = new EventEmitter();
	private socket: IUdpSocket<object> | null = null;

	constructor(
		private id: string,
		private udpSocketFactory: () => Promise<IUdpSocket<object>>,
	) {
		this.handleMessage = this.handleMessage.bind(this);
	}

	public setId(id: string) {
		this.id = id;
	}

	public async start() {
		if (this.socket) {
			throw new Error('PeerDiscoveryServiceMessageSender already started');
		}

		this.socket = await this.udpSocketFactory();
		this.socket.addListener(UdpSocketEvent.Message, this.handleMessage);
	}

	public async stop() {
		if (!this.socket) {
			throw new Error('PeerDiscoveryServiceMessageSender already stopped');
		}

		await this.socket.close();
		this.socket.removeListener(UdpSocketEvent.Message, this.handleMessage);

		this.socket = null;
	}

	public isStarted() {
		return Boolean(this.socket);
	}

	public onQuery(callback: (message: QueryMessage) => void) {
		this.emitter.on(Event.Query, callback);
	}

	public sendQuery() {
		return this.sendMessage<QueryMessage>({
			type: MessageType.Query,
		});
	}

	public onAnnounce(callback: (message: AnnounceMessage) => void) {
		this.emitter.on(Event.Announce, callback);
	}

	public sendAnnounce(address: string, port: number) {
		return this.sendMessage<AnnounceMessage>({
			type: MessageType.Announce,
			address,
			port,
		});
	}

	public onRenounce(callback: (message: RenounceMessage) => void) {
		this.emitter.on(Event.Renounce, callback);
	}

	public sendRenounce() {
		return this.sendMessage<RenounceMessage>({
			type: MessageType.Renounce,
		});
	}

	public onPing(callback: (message: PingMessage) => void) {
		this.emitter.on(Event.Ping, callback);
	}

	public sendPing() {
		return this.sendMessage<PingMessage>({
			type: MessageType.Ping,
		});
	}

	private handleMessage(message: unknown) {
		if (message && isBaseMessage(message)) {
			if (message.sourceId === this.id) {
				// ignore own message that come back via broadcast
				return;
			}

			debug('received message', JSON.stringify(message));

			switch (message.type) {
				case MessageType.Query:
					this.emitter.emit(Event.Query, message);
					break;
				case MessageType.Announce:
					this.emitter.emit(Event.Announce, message);
					break;
				case MessageType.Renounce:
					this.emitter.emit(Event.Renounce, message);
					break;
				case MessageType.Ping:
					this.emitter.emit(Event.Ping, message);
					break;
				default:
					console.warn(`UDPPeerDiscoveryService: Invalid message ${JSON.stringify(message)}`);
			}
		} else {
			console.warn(`UDPPeerDiscoveryService: Invalid message ${JSON.stringify(message)}`);
		}
	}

	private async sendMessage<TMessage extends BaseMessage>(contents: Omit<TMessage, 'sourceId'>) {
		if (!this.socket) {
			throw new Error("Trying to send message but PeerDiscoveryServiceMessageSender didn't start");
		}

		const message = {
			sourceId: this.id,
			...contents,
		};

		debug('send message', JSON.stringify(message));
		await this.socket.send(message);
	}
}
