import { PowerActionRule, PowerActionRules } from './IPowerActionTimer';
import IPowerActionTimerStorage from './IPowerActionTimerStorage';
import PowerActionTimerPropertyStorage from './PowerActionTimerPropertyStorage';

/**
 * Power Action Timer storage that copies rules from ``oldStorage`` to ``newStorage``
 * if ``newStorage`` hasn't been written to yet.
 *
 * Migration is ensured to run at most once before any method returns.
 */
export default class MigratingPowerActionTimerPropertyStorage implements IPowerActionTimerStorage {
	/**
	 * We want the migration done before any accessor runs and *only once*.
	 * We don't care if it's lazy or eager.
	 */
	private readonly migration: Promise<void> = this.performMigration(); // TODO: this is possibly uncaught exception

	constructor(
		private readonly oldStorage: () => PowerActionTimerPropertyStorage,
		private readonly newStorage: PowerActionTimerPropertyStorage,
	) {}

	public async put(uid: string, rule: PowerActionRule): Promise<void> {
		await this.migration;
		return await this.newStorage.put(uid, rule);
	}

	public async remove(uid: string): Promise<boolean> {
		await this.migration;
		return await this.newStorage.remove(uid);
	}

	public async clear(): Promise<boolean> {
		await this.migration;
		return await this.newStorage.clear();
	}

	public async content(): Promise<PowerActionRules> {
		await this.migration;
		return await this.newStorage.content();
	}

	private async performMigration() {
		if (!(await this.newStorage.hasPersistedData())) {
			await this.newStorage.putAll(await this.oldStorage().content());
			console.debug('Power Action Timer storage migration complete.');
		}
	}
}
