import IPowerActionTimer, { PowerActionRule, PowerActionRules } from './IPowerActionTimer';
import IPowerActionTimerStorage from './IPowerActionTimerStorage';
import IPowerActionScheduler, { PowerActionEvent } from './IPowerActionScheduler';
import IPowerActionExecutor from './IPowerActionExecutor';

export default class DefaultPowerActionTimer implements IPowerActionTimer {
	private started: boolean = false;

	constructor(
		private storage: IPowerActionTimerStorage,
		private scheduler: IPowerActionScheduler,
		private executor: IPowerActionExecutor,
	) {}

	public async set(uid: string, rule: PowerActionRule): Promise<void> {
		await this.storage.put(uid, rule);
		if (this.started) {
			this.scheduler.reschedule(await this.rules());
		}
	}

	public async unset(...uid: string[]): Promise<void> {
		let changed = false;
		await Promise.all(
			uid.map(async (u: string) => {
				changed = changed || (await this.storage.remove(u));
			}),
		);
		if (this.started && changed) {
			this.scheduler.reschedule(await this.rules());
		}
	}

	public async reset(): Promise<void> {
		const changed = await this.storage.clear();
		if (this.started && changed) {
			this.scheduler.reschedule(await this.rules());
		}
	}

	public async rules(): Promise<PowerActionRules> {
		return await this.storage.content();
	}

	public async notifyStart(): Promise<void> {
		this.started = true;
		this.scheduler.addListener(this.listener);
		this.scheduler.reschedule(await this.rules());
	}

	public async notifyEnd(): Promise<void> {
		this.started = false;
		this.scheduler.removeListener(this.listener);
		this.scheduler.unschedule();
	}

	private readonly listener = (event: PowerActionEvent) => {
		this.executor.execute(event);
	};
}
