import Weekday from '@signageos/actions/dist/Misc/Weekday';
import { CronJob } from 'cron';
import IPowerActionScheduler, { PowerActionListener } from './IPowerActionScheduler';
import { PowerActionRules } from './IPowerActionTimer';
import { EventEmitter } from 'events';

/**
 * Power Action Scheduler backed by ``CronJob``s which emits ``PowerActionEvent``s.
 */
export default class CronPowerActionScheduler implements IPowerActionScheduler {
	private readonly cronJobs: { [uid: string]: CronJob } = {};
	private readonly events: EventEmitter = new EventEmitter();

	constructor() {
		this.events.setMaxListeners(1);
	}

	public addListener(listener: PowerActionListener) {
		this.events.addListener('power', listener);
	}

	public removeListener(listener: PowerActionListener) {
		this.events.removeListener('power', listener);
	}

	public reschedule(rules: PowerActionRules): void {
		this.unschedule();
		this.schedule(rules);
	}

	public unschedule(): void {
		Object.keys(this.cronJobs).forEach((uid: string) => {
			this.cronJobs[uid].stop();
			delete this.cronJobs[uid];
		});
	}

	private schedule(rules: PowerActionRules): void {
		Object.keys(rules).forEach((uid: string) => {
			const { powerType, weekdays, time } = rules[uid];
			const cronJob = createCronJob(weekdays, time, () => {
				this.events.emit('power', powerType);
			});
			this.cronJobs[uid] = cronJob;
		});

		Object.keys(this.cronJobs).forEach((uid: string) => {
			this.cronJobs[uid].start();
		});
	}
}

function createCronJob(weekdays: Weekday[], time: string, onTick: () => void): CronJob {
	const weekdaysString = weekdays.join(',');
	const [, hours, minutes, seconds] = (time.match(/^([0-9]{2}):([0-9]{2})(?::([0-9]{2}))?$/) || []) as string[];
	return new CronJob(`${seconds ?? '00'} ${minutes} ${hours} * * ${weekdaysString}`, onTick);
}
