type IAddressObject = {
	address: string;
	priority: number;
};

export async function getLocalIpAddresses(timeoutMs: number = 5e3) {
	return new Promise((resolve: (addresses: string[]) => void) => {
		const RTCPeerConnection =
			/*window.RTCPeerConnection ||*/ (window as any).webkitRTCPeerConnection || (window as any).mozRTCPeerConnection;

		if (RTCPeerConnection) {
			const allAddresses: IAddressObject[] = [];

			(function () {
				const rtc = new RTCPeerConnection({ iceServers: [] });
				rtc.createDataChannel('', {
					reliable: false,
				});

				rtc.onicecandidate = function (evt: any) {
					// convert the candidate to SDP so we can run it through our general parser
					// see https://twitter.com/lancestout/status/525796175425720320 for details
					if (evt.candidate) {
						grepSDP('a=' + evt.candidate.candidate);
					}
				};
				rtc.createOffer(
					(offerDesc: any) => {
						grepSDP(offerDesc.sdp);
						rtc.setLocalDescription(offerDesc);
					},
					(error: Error) => {
						console.warn('offer failed', error);
					},
				);

				function grepSDP(sdp: any) {
					const addressesCandidates = sdp.split('\r\n').map((line: string) => {
						// c.f. http://tools.ietf.org/html/rfc4566#page-39
						if (line.indexOf('a=candidate') !== -1) {
							// http://tools.ietf.org/html/rfc4566#section-5.13
							const parts = line.split(' '); // http://tools.ietf.org/html/rfc5245#section-15.1
							const address = parts[4];
							const type = parts[7];
							if (type === 'host') {
								const priority = isLocalIpAddress(address) ? 5 : 3;
								return { priority, address };
							} else {
								return undefined;
							}
						} else if (line.indexOf('c=') !== -1) {
							// http://tools.ietf.org/html/rfc4566#section-5.7
							const parts = line.split(' ');
							const address = parts[2];
							const priority = isLocalIpAddress(address) ? 4 : 2;
							return { priority, address };
						} else {
							return undefined;
						}
					});
					const validatedAddresses = addressesCandidates.filter((addr: string | undefined) => addr);
					allAddresses.push(...validatedAddresses);
				}
			})();
			setTimeout(
				() =>
					resolve(
						allAddresses
							.sort((a: IAddressObject, b: IAddressObject) => b.priority - a.priority)
							.map((addressObject: IAddressObject) => addressObject.address),
					),
				timeoutMs,
			);
		} else {
			resolve([]);
		}
	});
}

function isLocalIpAddress(address: string) {
	return address.startsWith('192.168.') || address.startsWith('10.') || address.startsWith('172.');
}
