import { EventEmitter } from 'events';
import { getOrientation, getVideoOrientation, setOrientation } from '../Screen/screenRotationStorageHelper';
import { ResolutionItem } from '../ResolutionList';
import VideoOrientation from '@signageos/common-types/dist/Device/Resolution/VideoOrientation';
import Orientation from '@signageos/common-types/dist/Device/Resolution/Orientation';

export default class DefaultSettingsManager {
	private static VOLUME_KEY: string = 'default.native_device.VOLUME';
	private static RESOLUTION_KEY: string = 'default.native_device.RESOLUTION_KEY';

	private eventEmitter: EventEmitter = new EventEmitter();

	constructor(private window: Window) {}

	public getVolume() {
		const volumeValue = this.window.localStorage.getItem(DefaultSettingsManager.VOLUME_KEY);
		return volumeValue !== null ? parseInt(volumeValue) : 100;
	}

	public setVolume(volume: number) {
		this.window.localStorage.setItem(DefaultSettingsManager.VOLUME_KEY, volume.toString());
		this.eventEmitter.emit('volume_change', volume);
	}

	public onVolumeChange(listener: (volume: number) => void) {
		this.eventEmitter.addListener('volume_change', listener);
	}

	public getOrientation() {
		return getOrientation(this.window.localStorage);
	}

	public getVideoOrientation() {
		return getVideoOrientation(this.window.localStorage);
	}

	public setOrientation(orientation: Orientation, videoOrientation?: VideoOrientation) {
		return setOrientation(this.window.localStorage, orientation, videoOrientation);
	}

	public setResolution(resolution: ResolutionItem) {
		this.window.localStorage.setItem(DefaultSettingsManager.RESOLUTION_KEY, JSON.stringify(resolution));
	}

	public getResolution() {
		const resolutionItemRaw = this.window.localStorage.getItem(DefaultSettingsManager.RESOLUTION_KEY);
		if (resolutionItemRaw) {
			try {
				return JSON.parse(resolutionItemRaw) as ResolutionItem;
			} catch (error) {
				console.error('Failed to parse resolution', error);
			}
		}
		const resolutionItem: ResolutionItem = {
			width: Math.round(this.window.innerWidth * this.window.devicePixelRatio),
			height: Math.round(this.window.innerHeight * this.window.devicePixelRatio),
		};
		return resolutionItem;
	}
}
