import Orientation from '@signageos/common-types/dist/Device/Resolution/Orientation';
import VideoOrientation from '@signageos/common-types/dist/Device/Resolution/VideoOrientation';
import { ShortWeekdayEnum } from '@signageos/common-types/dist/Device/Settings/DeviceSettings';
import { NativeTimerType } from '@signageos/common-types/dist/Device/Timer/NativeTimerType';
import TimerWeekday from '@signageos/lib/dist/Timer/Power/PowerTimerWeekday';
import { convertWeekday } from '@signageos/lib/dist/Timer/convertWeekday';
import wait from '@signageos/lib/dist/Timer/wait';
import html2canvas from 'html2canvas';
import Cookies from 'js-cookie';
import _ from 'lodash';
import ICache from '../../Cache/ICache';
import ProprietaryCache from '../../Cache/ProprietaryCache';
import { createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher } from '../../FileSystem/FileSystemWithReservedSpace';
import { RESERVED_SPACE_PERCENTAGE } from '../../FileSystem/fileSystemHelper';
import MockDeviceFirmwareVersionFactory from '../../Firmware/MockDeviceFirmwareVersion';
import MockWifi from '../../Hardware/MockWifi';
import { ProprietaryInput } from '../../Input/ProprietaryInput';
import MockNetwork from '../../Management/Device/Network/MockNetwork';
import MockMonitors from '../../Monitors/MockMonitors';
import createPropertyStorage from '../../Property/propertyStorage';
import { hideOverlay, isOverlayShown, showOverlay, updateOverlay } from '../../Screen/overlayHelper';
import MockProximitySensor from '../../Sensors/MockProximitySensor';
import IServletRunner from '../../Servlet/IServletRunner';
import NotImplementedServletRunner from '../../Servlet/NotImplementedServletRunner';
import AsyncStorage from '../../Storage/Async/AsyncStorage';
import { asynchronizeStorage } from '../../Storage/Async/asyncStorageHelper';
import CategorizedLocalStorage from '../../Storage/CategorizedLocalStorage';
import WebBrowserEmulatorSystemInfo from '../../SystemInfo/WebBrowserEmulatorSystemInfo';
import { IProprietaryTimerStorage } from '../../Timer/ITimerStorage';
import ProprietaryTimerPropertyStorage from '../../Timer/ProprietaryTimerPropertyStorage';
import IBatteryStatus from '../Battery/IBatteryStatus';
import { getNoBatteryStatus } from '../Battery/batteryHelper';
import { IScripts } from '../CustomScripts/IScripts';
import IWifi from '../Hardware/IWifi';
import { IAutoRecoveryConfiguration, IAutoRecoverySettings } from '../IAutoRecovery';
import IBasicDriver from '../IBasicDriver';
import ICacheDriver from '../ICacheDriver';
import { IDeviceFirmwareVersion } from '../IDeviceFirmwareVersion';
import IFileSystem from '../IFileSystem';
import IMonitors from '../IMonitors';
import { IPeerRecoveryConfiguration, IPeerRecoverySettings } from '../IPeerRecovery';
import ISystemInfo from '../ISystemInfo';
import { IInput } from '../Input/IInput';
import IManagementDriver, { INativeCommands, IPlatformSpecificAPIs, ISensors } from '../Management/IManagementDriver';
import ManagementCapability from '../Management/ManagementCapability';
import NotImplementedMdc from '../NativeCommand/MDC/NotImplementedMdc';
import INetwork from '../Network/INetwork';
import IProxy from '../Network/IProxy';
import { NotImplementedProxy } from '../Network/NotImplementedProxy';
import { ONLY_FULL_HD_RESOLUTION_LIST, ResolutionItem } from '../ResolutionList';
import IScreenRotationManager from '../Screen/IScreenRotationManager';
import EmulatorTimeManager from '../TimeManager/EmulatorTimeManager';
import ITimeManager from '../TimeManager/ITimeManager';
import NativeFromDriverTimer from '../Timer/ITimer';
import NumberTimerType, { numberToNativeTimerType } from '../Timer/TimerType';
import { IVpn } from '../VPN/IVpn';
import ActionTimer from './ActionTimer';
import DefaultSettingsManager from './DefaultSettingsManager';
import IDisplayManager from '../DisplayManager/IDisplayManager';
import MockDisplayManager from '../DisplayManager/MockDisplayManager';
import { ManagementCapabilities } from '@signageos/common-types/dist/Device/Capabilities/ManagementCapabilities';
import { SecretManager } from '../../Management/Secrets/SecretManager';
import SubtleCryptoSecretManager from '../../Management/Secrets/SubtleCryptoSecretManager';
import NotImplementedScripts from '../CustomScripts/NotImplementedScripts';
import { createManagementPowerExecutors, PowerExecutors } from '../../Management/Device/Power/powerExecutors';
import DefaultBrowserSecretStorage from '../../Management/Secrets/DefaultBrowserSecretStorage';

type InternalTimerType = {
	type: NativeTimerType;
	timeOn: string | null;
	timeOff: string | null;
	weekdays: `${ShortWeekdayEnum}`[];
	volume: number;
};

export class DefaultManagementDriver implements IManagementDriver {
	private static BRIGHTNESS_KEY: string = 'default.native_device.BRIGHTNESS';
	private static TIMERS_KEY: string = 'default.native_device.TIMERS';
	private static AUTO_RECOVERY_KEY: string = 'default.native_device.AUTO_RECOVERY';
	private static AUTO_RECOVERY_DEFAULT_HEALTHCHECK_INTERVAL_MS: number = 30e3;
	private static PEER_RECOVERY_KEY: string = 'default.native_device.PEER_RECOVERY';
	private static EXTENDED_MANAGEMENT_URL_KEY: string = 'default.native_device.EXTENDED_MANAGEMENT_URL';
	private static REMOTE_CONTROL_ENABLED_KEY: string = 'default.native_device.REMOTE_CONTROL_ENABLED';
	private static HARDWARE_ACCELERATION_KEY: string = 'default.native_device.HARDWARE_ACCELERATION';

	public readonly input: IInput;
	public readonly proxy: IProxy = new NotImplementedProxy();
	public readonly systemInfo: ISystemInfo;
	public readonly sensors: ISensors;
	public readonly platformSpecificAPIs: IPlatformSpecificAPIs = {};
	public readonly monitors: IMonitors;
	public readonly network: INetwork;
	public readonly wifi: IWifi;
	public readonly vpn: IVpn;
	public readonly fileSystem: IFileSystem;
	public readonly servletRunner: IServletRunner;
	public readonly timeManager: ITimeManager;
	public readonly proprietaryTimerStorage: IProprietaryTimerStorage;
	public readonly nativeCommands: INativeCommands;
	public readonly scripts: IScripts;
	public readonly displayManager: IDisplayManager;
	public readonly secretManager: SecretManager;
	public readonly powerExecutors: PowerExecutors;

	private storage: Storage;
	private asyncStorage: AsyncStorage;
	private cache: ICache;
	private fakeFirmware?: IDeviceFirmwareVersion;
	private internalActionTimers: { autoRecovery: ActionTimer; peerRecovery: ActionTimer };

	protected settingsManager: DefaultSettingsManager;
	protected screenRotationManager: IScreenRotationManager;

	constructor(
		private window: Window,
		private deviceUid: string,
		settingsManager: DefaultSettingsManager,
		screenRotationManager: IScreenRotationManager,
		private cacheDriver: ICacheDriver,
		private basicDriver: IBasicDriver,
		private realFileSystem: IFileSystem & { initialize(): Promise<void> },
	) {
		const DEFAULT_TOTAL_SIZE_BYTES = 5 * 1024 * 1024; // Default quota of localStorage in browsers
		this.input = new ProprietaryInput(this.window);
		this.storage = new CategorizedLocalStorage(this.window.localStorage, deviceUid);
		this.asyncStorage = asynchronizeStorage(this.storage);
		this.cache = new ProprietaryCache(this.storage, DEFAULT_TOTAL_SIZE_BYTES);
		this.fileSystem = createFileSystemWithReservedSpaceWithWindowHttpHeadFetcher(
			this.realFileSystem,
			this.window,
			RESERVED_SPACE_PERCENTAGE,
		);
		this.settingsManager = settingsManager;
		this.screenRotationManager = screenRotationManager;
		this.servletRunner = new NotImplementedServletRunner();
		this.sensors = {
			proximity: new MockProximitySensor(this.window),
		};
		this.monitors = new MockMonitors(this.window);
		this.network = new MockNetwork(this.window);
		this.wifi = new MockWifi();
		this.systemInfo = new WebBrowserEmulatorSystemInfo(this.window);
		this.timeManager = new EmulatorTimeManager(this.window);
		this.proprietaryTimerStorage = new ProprietaryTimerPropertyStorage(createPropertyStorage(() => this.cacheDriver));
		this.internalActionTimers = {
			autoRecovery: new ActionTimer(this.asyncStorage, 'autoRecovery', async () => {
				const storedRaw = await this.asyncStorage.getItem(DefaultManagementDriver.AUTO_RECOVERY_KEY);
				const healthcheckIntervalMs = storedRaw
					? JSON.parse(storedRaw).healthcheckIntervalMs
					: DefaultManagementDriver.AUTO_RECOVERY_DEFAULT_HEALTHCHECK_INTERVAL_MS;

				console.log(`Device auto recovery automatically enabled with healthcheck interval ${healthcheckIntervalMs}ms.`);
				await this.setAutoRecovery({ enabled: true, healthcheckIntervalMs });
			}),
			peerRecovery: new ActionTimer(this.asyncStorage, 'peerRecovery', async () => {
				const storedRaw = await this.asyncStorage.getItem(DefaultManagementDriver.PEER_RECOVERY_KEY);
				const urlLauncherAddress = storedRaw ? JSON.parse(storedRaw).urlLauncherAddress : this.getConfigurationBaseUrl();

				console.log(`Device peer recovery automatically enabled with URL launcher address ${urlLauncherAddress}.`);
				await this.setPeerRecovery({ enabled: true, urlLauncherAddress });
			}),
		};
		this.nativeCommands = {
			mdc: new NotImplementedMdc(),
		};
		this.scripts = new NotImplementedScripts();
		this.displayManager = new MockDisplayManager();
		const defaultSecretStorage = new DefaultBrowserSecretStorage(this.window.localStorage, () => this.getDeviceUid());
		this.secretManager = new SubtleCryptoSecretManager(() => this.window.crypto.subtle, defaultSecretStorage);
		this.powerExecutors = createManagementPowerExecutors(this, cacheDriver);
	}

	public async isFrontEnabled(): Promise<boolean> {
		return true;
	}

	public async getAutoRecovery(): Promise<IAutoRecoverySettings> {
		const storedRaw = await this.asyncStorage.getItem(DefaultManagementDriver.AUTO_RECOVERY_KEY);
		if (storedRaw) {
			return JSON.parse(storedRaw);
		}
		return { enabled: false, autoEnableTimeoutMs: 0 };
	}

	public async setAutoRecovery(configuration: IAutoRecoveryConfiguration) {
		console.log(
			`Device auto recovery ${
				configuration.enabled ? `enabled with healthcheck interval ${configuration.healthcheckIntervalMs}ms` : 'disabled'
			}.`,
		);

		await this.mergeAsyncStorageItem(DefaultManagementDriver.AUTO_RECOVERY_KEY, configuration);

		if (configuration.enabled === false && configuration.autoEnableTimeoutMs) {
			await this.internalActionTimers.autoRecovery.set(configuration.autoEnableTimeoutMs);
		} else {
			await this.internalActionTimers.autoRecovery.drop();
		}
	}

	public async getPeerRecovery(): Promise<IPeerRecoverySettings> {
		const storedRaw = await this.asyncStorage.getItem(DefaultManagementDriver.PEER_RECOVERY_KEY);
		if (storedRaw) {
			return JSON.parse(storedRaw);
		}
		return { enabled: false, autoEnableTimeoutMs: 0 };
	}

	public async setPeerRecovery(configuration: IPeerRecoveryConfiguration) {
		console.log(
			`Device peer recovery ${
				configuration.enabled ? `enabled with URL launcher address ${configuration.urlLauncherAddress}` : 'disabled'
			}`,
		);

		await this.mergeAsyncStorageItem(DefaultManagementDriver.PEER_RECOVERY_KEY, configuration);

		if (configuration.enabled === false && configuration.autoEnableTimeoutMs) {
			await this.internalActionTimers.peerRecovery.set(configuration.autoEnableTimeoutMs);
		} else {
			await this.internalActionTimers.peerRecovery.drop();
		}
	}

	public async getConfigurationBaseUrl() {
		return this.basicDriver.getConfigurationBaseUrl();
	}

	public getApplicationType() {
		return this.basicDriver.getApplicationType();
	}

	public async managementSupports(capability: ManagementCapability) {
		switch (capability) {
			case ManagementCapability.SCREENSHOT_UPLOAD:
			case ManagementCapability.MODEL:
			case ManagementCapability.SERIAL_NUMBER:
			case ManagementCapability.BRAND:
			case ManagementCapability.OS_VERSION:
			case ManagementCapability.TIMERS_PROPRIETARY:
			case ManagementCapability.TIMERS_NATIVE:
			case ManagementCapability.STORAGE_UNITS:
			case ManagementCapability.SYSTEM_REBOOT:
			case ManagementCapability.APP_RESTART:
			case ManagementCapability.SET_VOLUME:
			case ManagementCapability.GET_VOLUME:
			case ManagementCapability.SET_TIME:
			case ManagementCapability.GET_TIMEZONE:
			case ManagementCapability.SET_BRIGHTNESS:
			case ManagementCapability.GET_BRIGHTNESS:
			case ManagementCapability.DISPLAY_POWER:
			case ManagementCapability.APP_UPGRADE:
			case ManagementCapability.SCREEN_RESIZE:
			case ManagementCapability.NETWORK_INFO:
			case ManagementCapability.WIFI:
			case ManagementCapability.WIFI_SCAN:
			case ManagementCapability.WIFI_AP:
			case ManagementCapability.PROXIMITY_SENSOR:
			case ManagementCapability.FACTORY_RESET:
			case ManagementCapability.FIRMWARE_UPGRADE:
			case ManagementCapability.ORIENTATION_LANDSCAPE:
			case ManagementCapability.ORIENTATION_PORTRAIT:
			case ManagementCapability.ORIENTATION_LANDSCAPE_FLIPPED:
			case ManagementCapability.ORIENTATION_PORTRAIT_FLIPPED:
			case ManagementCapability.SCHEDULE_POWER_ACTION:
			case ManagementCapability.AUTO_RECOVERY:
			case ManagementCapability.PEER_RECOVERY:
			case ManagementCapability.FILE_SYSTEM_WIPEOUT:
			case ManagementCapability.REMOTE_DESKTOP:
			case ManagementCapability.EXTENDED_MANAGEMENT:
			case ManagementCapability.CUSTOM_SCRIPTS:
			case ManagementCapability.SET_REMOTE_CONTROL_ENABLED:
			case ManagementCapability.DISPLAY_MANAGER:
			case ManagementCapability.HARDWARE_ACCELERATION:
			case ManagementCapabilities.SECRETS:
				return true;
			default:
				return false;
		}
	}

	public async initialize(_staticBaseUrl: string) {
		await updateOverlay(this.window, this.asyncStorage);
		await this.screenRotationManager.applyOrientation();
		this.updateBrightness();
		await this.realFileSystem.initialize();
		await this.internalActionTimers.autoRecovery.init();
		await this.internalActionTimers.peerRecovery.init();
	}

	public async systemReboot() {
		this.window.history.go(0);
	}

	public appRestart() {
		this.window.setTimeout(() => this.window.history.go(0), 500);
	}

	public async appUpgrade(baseUrl: string, version?: string) {
		return async () => {
			this.window.location.href =
				typeof version !== 'undefined' ? `${baseUrl}/app/default/${version}/index.html${this.window.location.search}` : baseUrl;
		};
	}

	public async packageInstall(_baseUrl: string, _packageName: string, _version: string, _build: string | null) {
		throw new Error('Not implemented package install');
	}

	public start() {
		return this.basicDriver.start();
	}

	public stop() {
		return this.basicDriver.stop();
	}

	public async displayIsPowerOn(): Promise<boolean> {
		return !(await isOverlayShown(this.asyncStorage));
	}

	public async displayPowerOn() {
		await hideOverlay(this.window, this.asyncStorage);
	}

	public async displayPowerOff() {
		await showOverlay(this.window, this.asyncStorage);
	}

	public async firmwareUpgrade(_baseUrl: string, _version?: string, _onProgress?: (progress: number) => void) {
		this.fakeFirmware = await new MockDeviceFirmwareVersionFactory().getFirmwareVersion(this.deviceUid, this.getApplicationType());

		const cachedUid = await this.getCachedFirmwareVersionKey();
		await this.cache.saveOne(cachedUid, this.fakeFirmware.firmware.uid);
		await wait(2e3);
		return () => this.systemReboot();
	}

	public async firmwareGetVersion() {
		let cachedUid;
		try {
			cachedUid = await this.getCachedFirmwareVersionKey();
			const fwVersionCached = await this.cache.fetchOne(cachedUid);
			if (fwVersionCached) {
				return fwVersionCached;
			}
		} catch (e) {
			console.warn(`${cachedUid} is not present in cache yet.`);
		}

		const FIRMWARE_VERSION_REGEX = /(Chrome|Chromium|Firefox)\/([0-9\.]+)/;
		const matches = this.window.navigator.userAgent.match(FIRMWARE_VERSION_REGEX);
		if (matches && matches[2]) {
			return `${matches[1]}-${matches[2]}`;
		} else {
			throw new Error(`Unsupported browser for emulator: ${this.window.navigator.userAgent}`);
		}
	}

	public async firmwareGetType() {
		return null;
	}

	public async getDeviceUid() {
		return this.basicDriver.getDeviceUid();
	}

	public async isConnected() {
		return this.basicDriver.isConnected();
	}

	public async screenResize(
		_baseUrl: string,
		orientation: Orientation,
		_resolution: ResolutionItem,
		_version: string,
		videoOrientation?: VideoOrientation,
	) {
		this.settingsManager.setOrientation(orientation, videoOrientation);
		await wait(2e3); // saving value to local storage isn't exactly synchronous so it needs time before application restart
		return async () => this.appRestart();
	}

	public async getCurrentOrientation() {
		return this.settingsManager.getOrientation();
	}

	public async setOrientation(_baseUrl: string, orientation: Orientation, videoOrientation?: VideoOrientation) {
		this.settingsManager.setOrientation(orientation, videoOrientation);
		await wait(2e3); // saving value to local storage isn't exactly synchronous so it needs time before application restart
		return async () => this.appRestart();
	}

	public async setResolution(resolution: ResolutionItem) {
		this.settingsManager.setResolution(resolution);
		await wait(2e3); // saving value to local storage isn't exactly synchronous so it needs time before application restart
		return async () => this.appRestart();
	}

	public async getResolution(): Promise<ResolutionItem> {
		return this.settingsManager.getResolution();
	}

	public async screenSetBrightness(_timeFrom1: string, brightness1: number, _timeFrom2: string, _brightness2: number) {
		this.window.localStorage.setItem(DefaultManagementDriver.BRIGHTNESS_KEY, brightness1.toString());
		this.updateBrightness();
	}

	public async screenGetBrightness() {
		const brightnessValue = this.window.localStorage.getItem(DefaultManagementDriver.BRIGHTNESS_KEY);
		const brightness = brightnessValue !== null ? parseInt(brightnessValue) : 100;
		return {
			timeFrom1: '00:00',
			brightness1: brightness,
			timeFrom2: '00:00',
			brightness2: brightness,
		};
	}

	public async getSessionId(sessionIdKey: string) {
		const duid = this.deviceUid;
		const sessionRaw = Cookies.get(sessionIdKey);
		try {
			const sessionIdMap = sessionRaw ? JSON.parse(sessionRaw) : {};
			return sessionIdMap[duid];
		} catch (e) {
			return undefined;
		}
	}

	public async setSessionId(sessionIdKey: string, sessionId: string) {
		const duid = this.deviceUid;
		const sessionRaw = Cookies.get(sessionIdKey);
		let sessionIdMap;
		try {
			sessionIdMap = sessionRaw ? JSON.parse(sessionRaw) : {};
		} catch (e) {
			sessionIdMap = {};
		}
		sessionIdMap[duid] = sessionId;
		Cookies.set(sessionIdKey, sessionIdMap, {
			expires: 10 * 365,
		});
	}

	public async getCurrentTemperature() {
		console.info(new Error('Not implemented get current temperature'));
		return 0;
	}

	public async getTimers(): Promise<NativeFromDriverTimer[]> {
		const timersJson = this.window.localStorage.getItem(DefaultManagementDriver.TIMERS_KEY);
		if (!timersJson) {
			return [];
		}
		const timers: { [timerType: string]: InternalTimerType } = JSON.parse(timersJson);
		return _.values(timers).map((timer: InternalTimerType) => ({
			type: NumberTimerType[timer.type as keyof typeof NativeTimerType],
			timeOn: timer.timeOn,
			timeOff: timer.timeOff,
			weekdays: convertWeekday.fromShort(timer.weekdays).toTimerWeekday(),
			volume: timer.volume,
		}));
	}

	public async setTimer(type: NumberTimerType, timeOn: string | null, timeOff: string | null, weekdays: TimerWeekday[], volume: number) {
		const timerKey = numberToNativeTimerType(type);
		const timersJson = this.window.localStorage.getItem(DefaultManagementDriver.TIMERS_KEY);
		const timers: { [timerType: string]: InternalTimerType } = timersJson ? JSON.parse(timersJson) : {};
		if (timeOn || timeOff) {
			timers[timerKey] = {
				type: timerKey,
				timeOn,
				timeOff,
				weekdays: convertWeekday.fromTimerWeekday(weekdays).toShort(),
				volume,
			};
		} else {
			delete timers[timerKey];
		}
		this.window.localStorage.setItem(DefaultManagementDriver.TIMERS_KEY, JSON.stringify(timers));
	}

	public timerSetOnOffTimeHoliday(_type: NativeTimerType, _onAtHoliday: boolean, _offAtHoliday: boolean) {
		console.info('Not implemented set on/off time holiday');
		return Promise.resolve();
	}

	public async remoteControlSetEnabled(enabled: boolean) {
		return this.window.localStorage.setItem(DefaultManagementDriver.REMOTE_CONTROL_ENABLED_KEY, enabled ? '1' : '0');
	}

	public async remoteControlIsEnabled() {
		return this.window.localStorage.getItem(DefaultManagementDriver.REMOTE_CONTROL_ENABLED_KEY) === '1' ? true : false;
	}

	public async screenshotUpload(uploadBaseUrl: string) {
		const uploadUri = uploadBaseUrl + '/upload/file?prefix=screenshot/';
		const screenElement = self.window.document.getElementsByTagName('iframe')[0]?.contentWindow?.document.body ?? self.window.document.body;
		const resolution = this.settingsManager.getResolution();
		const screenshotCanvas: HTMLCanvasElement = await html2canvas(screenElement, {
			width: resolution.width,
			height: resolution.height,
		});
		const screenshotBlob = await new Promise((resolve: (blob: Blob) => void) => screenshotCanvas.toBlob((blob) => resolve(blob!)));
		const formData = new FormData();
		formData.append('file', screenshotBlob);
		const response = await self.window.fetch(uploadUri, { method: 'POST', body: formData });
		if (!response.ok) {
			throw new Error('Error during upload screenshot request');
		}
		const data = await response.json();
		return {
			url: data.uri,
		};
	}

	public async setDebug(_enabled: boolean) {
		console.warn(new Error('Not implemented setDebug'));
	}

	public async isDebugEnabled() {
		return false;
	}

	public async getVolume(): Promise<number> {
		return this.settingsManager.getVolume();
	}

	public async setVolume(volume: number) {
		this.settingsManager.setVolume(volume);
	}

	public async batteryGetStatus(): Promise<IBatteryStatus> {
		if (this.window.navigator.getBattery) {
			const battery = await this.window.navigator.getBattery();
			return {
				percentage: battery.level * 100,
				chargeType: battery.charging ? 'AC' : 'NONE',
				isCharging: battery.charging,
				lastChargingTime: undefined!, // TODO
			};
		}
		return getNoBatteryStatus();
	}

	public resetSettings(): Promise<void> {
		return Promise.resolve(); // do nothing
	}

	public async factoryReset(): Promise<void> {
		// clear local storage
		this.storage.clear();
		// delete all files from persistent file storage
		const storageUnits = await this.fileSystem.listStorageUnits();
		for (let storageUnit of storageUnits) {
			const filesInRoot = await this.fileSystem.listFiles({ storageUnit, filePath: '' });
			for (let file of filesInRoot) {
				await this.fileSystem.deleteFile(file, true);
			}
		}
		await wait(1e3); // make sure that local storage clear takes effect
		this.appRestart();
	}

	public async getSupportedResolutions() {
		return ONLY_FULL_HD_RESOLUTION_LIST;
	}

	public async getExtendedManagementUrl(): Promise<string | null> {
		const value = this.window.localStorage.getItem(DefaultManagementDriver.EXTENDED_MANAGEMENT_URL_KEY);

		try {
			return new URL(value!).toString();
		} catch (e) {
			return null;
		}
	}

	public async setExtendedManagementUrl(urlInput: string | null) {
		try {
			const value = new URL(urlInput!).toString();
			this.window.localStorage.setItem(DefaultManagementDriver.EXTENDED_MANAGEMENT_URL_KEY, value);
		} catch {
			/* empty */
		}
	}

	public async setHardwareAcceleration(enabled: boolean) {
		this.window.localStorage.setItem(DefaultManagementDriver.HARDWARE_ACCELERATION_KEY, enabled ? '1' : '0');
	}

	public async isHardwareAccelerationEnabled() {
		return this.window.localStorage.getItem(DefaultManagementDriver.HARDWARE_ACCELERATION_KEY) === '1';
	}

	private updateBrightness() {
		const brightness = this.window.localStorage.getItem(DefaultManagementDriver.BRIGHTNESS_KEY);
		if (brightness !== null) {
			const newFilter = `brightness(${50 + parseInt(brightness) / 2}%)`;
			this.window.document.body.style.filter = newFilter;
			this.window.document.body.style.webkitFilter = newFilter;
		}
	}

	private async getCachedFirmwareVersionKey() {
		return `LAST_UPGRADED_FIRMWARE;${await this.getDeviceUid()}`;
	}

	private async mergeAsyncStorageItem<TContent extends object>(key: string, newContent: TContent) {
		const originalRawContent = await this.asyncStorage.getItem(key);
		const originalContent = originalRawContent ? JSON.parse(originalRawContent) : {};

		await this.asyncStorage.setItem(key, JSON.stringify({ ...originalContent, ...newContent }));
	}
}
