import { all } from 'redux-saga/effects';
import startApplicationSaga from '../Application/startApplicationSaga';
import {
	afterRegistrationDeviceUidAuthentication,
	autoDeviceAuthentication,
	fallbackDeviceUidAuthentication,
	storeDeviceSession,
} from '../Authentication/authenticationSagas';
import { dependencyInjection } from '../DI/dependenciesSaga';
import { debugAllActions } from '../Debug/debugSagas';
import { updatingConfiguration } from '../Device/Configuration/deviceConfigurationSagas';
import { notifyDeviceStatus } from '../Device/Status/deviceAliveSaga';
import { deviceVerification } from '../Device/Verification/deviceVerificationSagas';
import { fallbackDeviceRegistration, notifyDeviceFrontDisplayVersion } from '../Device/deviceSagas';
import IConfig from '../Display/IConfig';
import { IResponsibilities } from '../Feature/Responsibilities';
import { IBundledApplet } from '../Front/Applet/BundledApplet/bundledAppletActions';
import { displayTimerChecking } from '../Front/Device/Timer/timerControllerSagas';
import { deviceSchedulePowerAction } from './Device/Power/deviceSchedulePowerActionSagas';
import { offlineActionsSaga } from '../Offline/offlineActionsSagas';
import OfflineCache from '../OfflineCache/OfflineCache';
import { IPropertyStorage } from '../Property/propertyStorage';
import { remoteDesktopSaga } from '../RemoteDesktop/remoteDesktopSaga';
import { healthChecking } from '../Socket/socketHealthCheckSagas';
import { socketPlatformCreate } from '../Socket/socketSagas';
import { commonSystemLogsSaga } from '../SystemLogs/commonSystemLogsSagas';
import * as TestFramework from '../Test/TestFramework';
import { runTestsSaga } from '../Test/testSagas';
import { pinging } from './Application/pingSagas';
import { syncApplicationVersionSaga } from './Device/Application/deviceApplicationSagas';
import { deviceVolumeSaga } from './Device/Audio/deviceVolumeSagas';
import { autoRecoverySaga } from './Device/AutoRecovery/autoRecoverySagas';
import { deviceBatterySaga } from './Device/Battery/deviceBatterySaga';
import { brightnessRefresh } from './Device/Brightness/brightnessControllerSagas';
import { syncBrightnessSettingsSaga } from './Device/Brightness/deviceBrightnessSagas';
import { startupDeviceDateTimeSettings, updateTimeSettingsSaga } from './Device/DateTime/currentTimeSyncSagas';
import { devicePerformDebugSettings } from './Device/Debug/deviceDebugSettingsSagas';
import { deviceDeprovision } from './Device/Deprovision/deviceDeprovisionSagas';
import { deviceFirmwareNotify, deviceFirmwareUpgrade } from './Device/Firmware/deviceFirmwareSagas';
import { installPackageSaga } from './Device/Package/devicePackageSagas';
import { peerRecoverySaga } from './Device/PeerRecovery/peerRecoverySaga';
import { devicePolicySaga } from './Device/Policy/devicePolicySaga';
import { subscribeDevicePolicy, updateDevicePolicy } from './Device/Policy/devicePolicySagas';
import {
	deviceDisplayPowerAction,
	devicePerformPowerAction,
	devicePerformScheduledPowerAction,
} from './Device/Power/devicePowerActionSagas';
import { syncRemoteControlSettingsSaga } from './Device/RemoteControl/deviceRemoteControlSagas';
import { reportScreen, resizeScreen } from './Device/Screen/deviceScreenResizeSagas';
import { createDeviceStorageSaga } from './Device/Storage/deviceStorageSaga';
import { calculateInitialTelemetryTimeoutMs } from './Device/Telemetry/deviceTelemetryHelper';
import { deviceTelemetrySaga } from './Device/Telemetry/deviceTelemetrySaga';
import { deviceTemperatureSaga } from './Device/Temperature/deviceTemperatureSaga';
import { syncDeviceTimerSettingsSaga } from './Device/Timer/deviceTimerSettingsSagas';
import { notifyDeviceApplicationVersion, notifyDeviceSaga } from './Device/deviceSagas';
import { deviceExtendedManagementSaga } from './Extended/extendedManagementSagas';
import { instantScreenshotSaga, periodicScreenshotSaga } from './Screen/screenshotSaga';
import IBundledServlet from './Servlet/IBundledServlet';
import { runBundledServlet, runServlets, subscribeActiveServlets } from './Servlet/servletSagas';
import { systemLogsSaga as managementSystemLogsSaga } from './SystemLogs/systemLogsSagas';
import AudioTest from './Tests/AudioTest';
import BatteryTest from './Tests/BatteryTest';
import DebugTest from './Tests/DebugTest';
import DisplayTest from './Tests/DisplayTest';
import InfoTest from './Tests/InfoTest';
import { IManagementState } from './managementReducers';
import { vpnSaga } from '../VPN/vpnSaga';
import { createProprietaryTimerManagerForManagement } from '../Timer/timerResolverFactory';
import { checkingStatusChange } from '../Front/Network/networkSagas';
import { customScriptsSaga } from '../CustomScripts/customScriptsSaga';
import { ManagementCacheDriver, normalizeManagementCacheDriver } from '../NativeDevice/Default/combinedDriver';
import { displayManagerSaga } from './Device/DisplayManager/displayManagerSaga';
import { cryptographySaga } from './Device/Secrets/deviceCryptographicSaga';

export interface IManagementSagaOptions {
	nativeDriver: ManagementCacheDriver;
	publicKey: string;
	sessionIdKey: string;
	frontDisplayVersion: string;
	autoVerification: { organizationUid: string; deviceName?: string } | undefined;
	applicationVersion: string;
	bundledApplet: IBundledApplet | null;
	bundledServlet: IBundledServlet | null;
	responsibilities: IResponsibilities;
	/** Additional tests that should run, apart from the standard suite that always runs */
	managementExtraTests?: Promise<TestFramework.Describe>[];
	logOfflineActions?: boolean;
	pingUrl: string;
}

export function* managementSaga(
	defaultConfig: IConfig,
	{
		nativeDriver,
		publicKey,
		sessionIdKey,
		frontDisplayVersion,
		autoVerification,
		applicationVersion,
		bundledApplet,
		bundledServlet,
		responsibilities,
		managementExtraTests,
		logOfflineActions = true,
		pingUrl,
	}: IManagementSagaOptions,
	getState: () => IManagementState,
	propertyStorage: IPropertyStorage,
) {
	const { managementDriver, cacheDriver } = normalizeManagementCacheDriver(nativeDriver);

	const offlineCache = new OfflineCache(managementDriver.fileSystem);
	const getNativeDriver = () => nativeDriver;
	const getManagementDriver = () => managementDriver;
	const proprietaryTimerStorage = managementDriver.proprietaryTimerStorage;
	const proprietaryTimerManager = createProprietaryTimerManagerForManagement(responsibilities, proprietaryTimerStorage, () =>
		managementDriver.displayIsPowerOn(),
	);
	const { powerActionTimer, powerKernel, powerExecutorFacade } = managementDriver.powerExecutors;

	yield all([
		// create DI container as first
		dependencyInjection({
			applicationVersion,
			managementDriver,
			cacheDriver,
			propertyStorage,
			responsibilities,
			proprietaryTimerStorage,
			bundledApplet,
			powerActionTimer,
			telemetryInitialTimeoutMs: calculateInitialTelemetryTimeoutMs(),
			pingUrl,
		}),
		// shared sagas
		socketPlatformCreate(
			getManagementDriver,
			'management',
			() => managementDriver.timeManager.getEpochMillis(),
			defaultConfig,
			powerKernel,
		), // Must be first saga
		healthChecking(),
		updatingConfiguration(getManagementDriver, getManagementDriver, propertyStorage, defaultConfig, publicKey, 'management'),
		autoDeviceAuthentication(sessionIdKey, getManagementDriver),
		storeDeviceSession(sessionIdKey, getManagementDriver),
		fallbackDeviceUidAuthentication(getManagementDriver),
		afterRegistrationDeviceUidAuthentication(getManagementDriver),
		fallbackDeviceRegistration(getManagementDriver, autoVerification),
		deviceVerification(getManagementDriver),
		// management sagas
		syncApplicationVersionSaga(responsibilities),
		brightnessRefresh(getManagementDriver, propertyStorage),
		deviceDisplayPowerAction(getManagementDriver),
		devicePerformScheduledPowerAction(powerExecutorFacade),
		deviceSchedulePowerAction(getManagementDriver, responsibilities, powerActionTimer),
		displayTimerChecking(responsibilities, proprietaryTimerManager),
		startupDeviceDateTimeSettings(),
		notifyDeviceSaga(getManagementDriver, responsibilities),
		notifyDeviceFrontDisplayVersion(frontDisplayVersion),
		notifyDeviceApplicationVersion(getManagementDriver, responsibilities, applicationVersion),
		pinging(),
		deviceFirmwareNotify(responsibilities, () => managementDriver.getDeviceUid()),
		deviceFirmwareUpgrade(responsibilities, () => managementDriver.getDeviceUid()),
		deviceVolumeSaga(getManagementDriver, responsibilities),
		syncBrightnessSettingsSaga(getManagementDriver, responsibilities, propertyStorage),
		installPackageSaga(responsibilities),
		deviceDeprovision(getNativeDriver, responsibilities, sessionIdKey),
		devicePerformPowerAction(getManagementDriver, responsibilities, powerExecutorFacade),
		syncDeviceTimerSettingsSaga(getManagementDriver, responsibilities, proprietaryTimerStorage),
		resizeScreen(responsibilities),
		reportScreen(responsibilities),
		syncRemoteControlSettingsSaga(getManagementDriver, responsibilities),
		deviceExtendedManagementSaga(getManagementDriver, responsibilities),
		updateTimeSettingsSaga(),
		devicePerformDebugSettings(getManagementDriver, responsibilities),
		deviceBatterySaga()(),
		createDeviceStorageSaga()(),
		deviceTemperatureSaga(responsibilities),
		periodicScreenshotSaga(responsibilities),
		instantScreenshotSaga(responsibilities),
		runTestsSaga([
			AudioTest(managementDriver),
			BatteryTest(managementDriver),
			DebugTest(managementDriver),
			DisplayTest(managementDriver, responsibilities),
			InfoTest(managementDriver),
			...(managementExtraTests ?? []),
		]),
		subscribeActiveServlets(getState, responsibilities, getManagementDriver),
		commonSystemLogsSaga(),
		managementSystemLogsSaga(),
		...(logOfflineActions ? [offlineActionsSaga('management.', getManagementDriver, getManagementDriver)] : []),
		runServlets(responsibilities, offlineCache),
		runBundledServlet(responsibilities, bundledServlet),
		deviceTelemetrySaga(
			responsibilities,
			getManagementDriver,
			propertyStorage,
			proprietaryTimerStorage,
			applicationVersion,
			powerActionTimer,
		)(),
		devicePolicySaga(),
		subscribeDevicePolicy(responsibilities, getState),
		updateDevicePolicy(responsibilities, propertyStorage),
		remoteDesktopSaga(),
		debugAllActions(),
		startApplicationSaga(),
		notifyDeviceStatus(),
		autoRecoverySaga(getManagementDriver, responsibilities),
		peerRecoverySaga(getManagementDriver, responsibilities),
		vpnSaga(getManagementDriver, responsibilities),
		checkingStatusChange(getManagementDriver),
		customScriptsSaga(offlineCache),
		displayManagerSaga(),
		cryptographySaga(getManagementDriver, responsibilities),
	]);
}
