import * as t from './../../Test/TestFramework';
import IManagementDriver from '../../NativeDevice/Management/IManagementDriver';
import should from '../../Test/should';
import ManagementCapability from '../../NativeDevice/Management/ManagementCapability';
import { TestCase } from '@signageos/actions/dist/Device/Test/deviceTestActions';

export default async (nativeDriver: IManagementDriver) => {
	return t.describe(TestCase.INFO, function* () {
		yield t.it('Should get serial number', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.SERIAL_NUMBER))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.SERIAL_NUMBER]}" capability`);
			}

			const sn = await nativeDriver.systemInfo.getSerialNumber();
			should(sn).ok();
			should(sn.length).be.greaterThanOrEqual(2);
		});

		yield t.it('Should get model', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.MODEL))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.MODEL]}" capability`);
			}

			const model = await nativeDriver.systemInfo.getModel();
			should(model).ok();
			should(model.length).be.greaterThanOrEqual(2);
		});

		yield t.it('Should get brand', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.BRAND))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.BRAND]}" capability`);
			}
			const brand = await nativeDriver.systemInfo.getBrand();
			should(brand).ok();
			should(brand.length).be.greaterThanOrEqual(2);
		});

		yield t.it('Should get firmware version', async () => {
			const fw = await nativeDriver.firmwareGetVersion();
			should(fw).be.ok();
		});

		yield t.it('Should get application type', async () => {
			const appType = await nativeDriver.getApplicationType();
			should(appType).be.ok();
		});

		yield t.it('Should get network info', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.NETWORK_INFO))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.NETWORK_INFO]}" capability`);
			}

			const networkInterfaces = await nativeDriver.network.listInterfaces();
			should(networkInterfaces).be.Array();
			should(networkInterfaces).be.not.empty();
			for (let networkInterface of networkInterfaces) {
				should(networkInterface.type).be.oneOf(['wifi', 'ethernet']);
				should.exist(networkInterface.macAddress);
				if (networkInterface.localAddress) {
					should.exist(networkInterface.netmask);
					should.exist(networkInterface.gateway);
				} else {
					should.not.exist(networkInterface.netmask);
					should.not.exist(networkInterface.gateway);
					// should.not.exist(networkInterface.dns);
				}
			}
		});

		yield t.it('Should get current temperature', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.TEMPERATURE))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.TEMPERATURE]}" capability`);
			}

			const temp = await nativeDriver.getCurrentTemperature();
			should(temp).be.greaterThan(0);
			should(temp).be.lessThan(999);
		});

		yield t.it('Should get current cpu usage', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.SYSTEM_CPU))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.SYSTEM_CPU]}" capability`);
			}
			const systemCpuUsage = await nativeDriver.systemInfo.getCpuUsage();
			should(systemCpuUsage).be.ok();
			should(systemCpuUsage).be.greaterThanOrEqual(0);
		});

		yield t.it('Should get current memory usage', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.SYSTEM_MEMORY))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.SYSTEM_MEMORY]}" capability`);
			}
			const systemMemoryUsage = await nativeDriver.systemInfo.getMemoryUsage();
			should(systemMemoryUsage).be.ok();
			should(systemMemoryUsage.free).be.greaterThanOrEqual(0);
			should(systemMemoryUsage.total).be.greaterThanOrEqual(0);
			should(systemMemoryUsage.used).be.greaterThanOrEqual(0);
		});

		yield t.it('Should get proxy connect status', async () => {
			if (!(await nativeDriver.managementSupports(ManagementCapability.PROXY))) {
				t.skip(`The device does not provide "${ManagementCapability[ManagementCapability.PROXY]}" capability`);
			}
			const proxyInfo = await nativeDriver.proxy.isEnabled();
			should(proxyInfo).is.Boolean();
			if (proxyInfo) {
				const proxyUri = await nativeDriver.proxy.getConnectedTo();
				should(proxyUri).not.be.undefined();
			}
		});
	});
};
