import { RsaKeyPairData, SecretStorage } from './SecretStorage';

export default class DefaultBrowserSecretStorage implements SecretStorage {
	private static readonly KEY_ALGORITHM: string = 'PBKDF2';
	private static readonly KEY_HASH: string = 'SHA-256';
	private static readonly KEY_LENGTH: number = 256;
	private static readonly KEY_ITERATIONS: number = 100_000;
	private static readonly AES_ALGORITHM: string = 'AES-GCM';
	private static readonly AES_IV_LENGTH: number = 12;
	private static readonly SALT_LENGTH: number = 16;

	constructor(
		private storage: Storage,
		private getDeviceUid: () => Promise<string>,
	) {}

	public async saveKeyPair(name: string, keyPair: RsaKeyPairData) {
		const password = await this.getDeviceUid();
		const salt = crypto.getRandomValues(new Uint8Array(DefaultBrowserSecretStorage.SALT_LENGTH));
		const key = await this.deriveKey(password, salt);
		const { encrypted, iv } = await this.encryptData(key, JSON.stringify(keyPair));
		this.storage.setItem(
			name,
			JSON.stringify({
				encrypted: this.uint8ToBase64(this.base64ToUint8(encrypted)),
				iv: this.uint8ToBase64(this.base64ToUint8(iv)),
				salt: this.uint8ToBase64(salt),
			}),
		);
	}

	public async getKeyPair(name: string): Promise<RsaKeyPairData | null> {
		const stringified = this.storage.getItem(name);
		if (!stringified) {
			return null;
		}
		try {
			const { encrypted, iv, salt } = JSON.parse(stringified);
			const password = await this.getDeviceUid();
			const saltArray = this.base64ToUint8(salt);
			const key = await this.deriveKey(password, saltArray);
			const decrypted = await this.decryptData(key, encrypted, iv);
			return JSON.parse(decrypted);
		} catch (e) {
			console.error('Failed to decrypt key pair:', e);
			return null;
		}
	}

	private async deriveKey(password: string, salt: Uint8Array): Promise<CryptoKey> {
		const baseKey = await crypto.subtle.importKey(
			'raw',
			new TextEncoder().encode(password),
			{ name: DefaultBrowserSecretStorage.KEY_ALGORITHM },
			false,
			['deriveKey'],
		);
		return crypto.subtle.deriveKey(
			{
				name: DefaultBrowserSecretStorage.KEY_ALGORITHM,
				salt,
				iterations: DefaultBrowserSecretStorage.KEY_ITERATIONS,
				hash: DefaultBrowserSecretStorage.KEY_HASH,
			},
			baseKey,
			{ name: DefaultBrowserSecretStorage.AES_ALGORITHM, length: DefaultBrowserSecretStorage.KEY_LENGTH },
			false,
			['encrypt', 'decrypt'],
		);
	}

	private async encryptData(key: CryptoKey, data: string): Promise<{ encrypted: string; iv: string }> {
		const iv = crypto.getRandomValues(new Uint8Array(DefaultBrowserSecretStorage.AES_IV_LENGTH));
		const encrypted = await crypto.subtle.encrypt(
			{ name: DefaultBrowserSecretStorage.AES_ALGORITHM, iv },
			key,
			new TextEncoder().encode(data),
		);
		return {
			encrypted: this.uint8ToBase64(new Uint8Array(encrypted)),
			iv: this.uint8ToBase64(iv),
		};
	}

	private async decryptData(key: CryptoKey, encrypted: string, ivBase64: string): Promise<string> {
		const encryptedBuffer = this.base64ToUint8(encrypted).buffer;
		const iv = this.base64ToUint8(ivBase64);
		const decrypted = await crypto.subtle.decrypt({ name: DefaultBrowserSecretStorage.AES_ALGORITHM, iv }, key, encryptedBuffer);
		return new TextDecoder().decode(decrypted);
	}

	private uint8ToBase64(arr: Uint8Array): string {
		if (typeof Buffer !== 'undefined') {
			return Buffer.from(arr).toString('base64');
		}
		let binary = '';
		for (let i = 0; i < arr.length; i++) {
			binary += String.fromCharCode(arr[i]);
		}
		return btoa(binary);
	}

	private base64ToUint8(base64: string): Uint8Array {
		if (typeof Buffer !== 'undefined') {
			return new Uint8Array(Buffer.from(base64, 'base64'));
		}
		const binary = atob(base64);
		const arr = new Uint8Array(binary.length);
		for (let i = 0; i < binary.length; i++) {
			arr[i] = binary.charCodeAt(i);
		}
		return arr;
	}
}
