import { fork, put, takeEvery } from 'redux-saga/effects';
import { TakeInstantScreenshot } from '@signageos/actions/dist/Device/Screenshot/deviceScreenshotActions';
import IManagementDriver from '../../NativeDevice/Management/IManagementDriver';
import ManagementCapability from '../../NativeDevice/Management/ManagementCapability';
import { SaveTakenScreenshot } from '@signageos/actions/dist/Screen/screenshotActions';
import { bindWhenPlatform } from '../../Socket/socketActionCreator';
import { IResponsibilities } from '../../Feature/Responsibilities';
import Responsibility from '../../Feature/Responsibility';
import { whenCapable } from '../../Feature/capable';
import { withDependencies } from '../../DI/dependencyInjection';
import { registerTelemetry } from '../Device/Telemetry/deviceTelemetryHelper';
import { isOnline } from '../../Util/isOnline';

const RECOVERING_DELAY = 30 * 60e3;

export function* periodicScreenshotSaga(responsibilities: IResponsibilities) {
	if (!responsibilities.has(Responsibility.SCREENSHOT)) {
		return;
	}

	yield fork(
		whenCapable(
			ManagementCapability.SCREENSHOT_UPLOAD,
			withDependencies(
				['managementDriver', 'uploadBaseUrl', 'telemetryIntervals', 'pingUrl', 'featureFlags'],
				function* ({ managementDriver, uploadBaseUrl, telemetryIntervals, pingUrl, featureFlags }) {
					const periodMs = telemetryIntervals?.screenshots ?? 0;
					let tryRecoverScreenshotsWhenFirewallBlocks = false;
					yield registerTelemetry('SCREENSHOT', periodMs, function* () {
						try {
							const onLine: boolean = yield isOnline(pingUrl);
							const isOnlineOrRecovering = onLine || tryRecoverScreenshotsWhenFirewallBlocks;
							const isScreenshotCaptureEnabled = featureFlags?.screenshotCapture ?? true;
							if (isOnlineOrRecovering && isScreenshotCaptureEnabled) {
								// Do not try if device is offline
								yield takeScreenshot(managementDriver, uploadBaseUrl, true);
							}
						} catch (error) {
							// In case of navigator.onLine is not working because of firewall, do try recovery after 30 minutes
							tryRecoverScreenshotsWhenFirewallBlocks = false;
							setTimeout(() => (tryRecoverScreenshotsWhenFirewallBlocks = true), RECOVERING_DELAY);
							console.error('screenshotSaga', error);
						}
					});
				},
			),
		),
	);
}

export function* instantScreenshotSaga(responsibilities: IResponsibilities) {
	if (!responsibilities.has(Responsibility.SCREENSHOT)) {
		return;
	}

	yield bindWhenPlatform(TakeInstantScreenshot);

	yield fork(
		whenCapable(
			ManagementCapability.SCREENSHOT_UPLOAD,
			withDependencies(
				['managementDriver', 'uploadBaseUrl', 'featureFlags'],
				function* ({ managementDriver, uploadBaseUrl, featureFlags }) {
					if (featureFlags?.screenshotCapture ?? true) {
						yield takeEvery(TakeInstantScreenshot, function* () {
							try {
								yield takeScreenshot(managementDriver, uploadBaseUrl, true);
							} catch (error) {
								console.error('instantScreenshotSaga', error);
							}
						});
					}
				},
			),
		),
	);
}

function* takeScreenshot(nativeDriver: IManagementDriver, uploadBaseUrl: string, computeHash: boolean) {
	const screenshotUri: { url: string; imageInformations?: { aHash?: string; dHash?: string } } = yield nativeDriver.screenshotUpload(
		uploadBaseUrl,
		computeHash,
	);
	yield put({
		type: SaveTakenScreenshot,
		screenshotUri: screenshotUri.url,
		imageHashes: { ...screenshotUri.imageInformations },
	} as SaveTakenScreenshot);
}
