import { takeEveryAndBindWhenPlatform } from '../../../Socket/socketActionCreator';
import { put, call } from 'redux-saga/effects';
import {
	SetDeviceTimerSettings,
	SetDeviceTimerSettingsSucceeded,
	SetDeviceTimerSettingsFailed,
	SetDeviceNativeTimerSettings,
} from '@signageos/actions/dist/Device/Timer/deviceTimerActions';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import { convertWeekday } from '@signageos/lib/dist/Timer/convertWeekday';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { setProprietaryTimer } from '../../../Device/Timer/deviceProprietaryTimerSagas';
import { IProprietaryTimerStorage } from '../../../Timer/ITimerStorage';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import { SagaIteratorWithPromises } from '../../../ReduxSaga/extendedEffects';
import { TimerLevel } from '@signageos/common-types/dist/Device/Timer/TimerLevel';
import NumberTimerType from '../../../NativeDevice/Timer/TimerType';

export function* setDeviceTimerSettingsCallback(
	getNativeDriver: () => IManagementDriver,
	proprietaryTimerStorage: IProprietaryTimerStorage,
	action: SetDeviceTimerSettings,
): SagaIteratorWithPromises {
	const nativeDriver = getNativeDriver();
	const supportsNativeTimers: boolean = yield nativeDriver.managementSupports(ManagementCapability.TIMERS_NATIVE);
	const supportsProprietaryTimers: boolean = yield nativeDriver.managementSupports(ManagementCapability.TIMERS_PROPRIETARY);
	if (supportsNativeTimers && action.level === TimerLevel.Native) {
		try {
			yield call(setNativeTimer, nativeDriver, action);
			yield put({
				type: SetDeviceTimerSettingsSucceeded,
				uid: action.uid,
			} as SetDeviceTimerSettingsSucceeded);
		} catch (error) {
			console.error('syncTimerSettingsSaga', error);
			yield put({
				type: SetDeviceTimerSettingsFailed,
				uid: action.uid,
			} as SetDeviceTimerSettingsFailed);
		}
	} else if (supportsProprietaryTimers) {
		// In addition to handling proprietary timers,
		// If native timers are not supported set proprietary timers instead
		try {
			yield call(setProprietaryTimer, proprietaryTimerStorage, action);
			yield put({
				type: SetDeviceTimerSettingsSucceeded,
				uid: action.uid,
			} as SetDeviceTimerSettingsSucceeded);
		} catch (error) {
			console.error('syncTimerSettingsSaga', error);
			yield put({
				type: SetDeviceTimerSettingsFailed,
				uid: action.uid,
			} as SetDeviceTimerSettingsFailed);
		}
	}
}

export function* syncDeviceTimerSettingsSaga(
	getNativeDriver: () => IManagementDriver,
	responsibilities: IResponsibilities,
	proprietaryTimerStorage: IProprietaryTimerStorage,
) {
	if (!responsibilities.has(Responsibility.TIMERS)) {
		return;
	}

	yield takeEveryAndBindWhenPlatform(SetDeviceTimerSettings, setDeviceTimerSettingsCallback, getNativeDriver, proprietaryTimerStorage);
}

export async function setNativeTimer(nativeDriver: IManagementDriver, action: SetDeviceNativeTimerSettings) {
	await nativeDriver.setTimer(
		NumberTimerType[action.timerType],
		action.timeOn,
		action.timeOff,
		convertWeekday.fromShort(action.weekdays).toTimerWeekday(),
		action.volume,
	);
}
