import { DeviceSettingsType } from '@signageos/common-types/dist/Device/Settings/DeviceSettingsType';
import { CheckAndSetResult, createDevicePolicySaga } from '../Policy/devicePolicySagas';
import IPolicyCheckAndSetProperties from '../Policy/IPolicyCheckAndSetProperties';
import _ from 'lodash';
import { getProprietaryTimers, getNativeTimers } from './deviceTimerTelemetrySagas';
import { ILongWeekdayTimer, IShortWeekdayTimer } from '@signageos/common-types/dist/Device/Settings/DeviceSettings';
import { convertWeekday } from '@signageos/lib/dist/Timer/convertWeekday';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import Responsibility from '../../../Feature/Responsibility';
import NumberTimerType from '../../../NativeDevice/Timer/TimerType';

export const deviceTimerPolicySaga = createDevicePolicySaga(
	DeviceSettingsType.TIMERS,
	Responsibility.TIMERS,
	ManagementCapability.TIMERS_NATIVE,
	checkAndSetTimersSettings,
);
export const deviceProprietaryTimerPolicySaga = createDevicePolicySaga(
	DeviceSettingsType.PROPRIETARY_TIMERS,
	Responsibility.TIMERS,
	ManagementCapability.TIMERS_PROPRIETARY,
	checkAndSetProprietaryTimersSettings,
);

export async function* checkAndSetTimersSettings(properties: IPolicyCheckAndSetProperties<DeviceSettingsType.TIMERS>) {
	const newTimersSettings = properties.policy.value;
	const currentTimersSettings = await getNativeTimers(properties);

	if (!_.isEqual(normalizeTimers(newTimersSettings), normalizeTimers(currentTimersSettings))) {
		for (let timerSettings of newTimersSettings) {
			await properties.managementDriver.setTimer(
				NumberTimerType[timerSettings.type as keyof typeof NumberTimerType],
				timerSettings.timeOn,
				timerSettings.timeOff,
				convertWeekday.fromLong(timerSettings.weekdays).toTimerWeekday(),
				timerSettings.volume,
			);
		}
		const timersToRemove = currentTimersSettings.filter(
			(currentTimer) => !newTimersSettings.some((newTimer) => currentTimer.type === newTimer.type),
		);
		for (const { type } of timersToRemove) {
			yield properties.managementDriver.setTimer(NumberTimerType[type as keyof typeof NumberTimerType], null, null, [], 0);
		}
		return CheckAndSetResult.CHANGED;
	}
	return CheckAndSetResult.NOT_CHANGED;
}

export async function* checkAndSetProprietaryTimersSettings(
	properties: IPolicyCheckAndSetProperties<DeviceSettingsType.PROPRIETARY_TIMERS>,
) {
	const newTimersSettings = properties.policy.value;
	const currentTimersSettings = await getProprietaryTimers(properties);

	if (!_.isEqual(normalizeTimers(newTimersSettings), normalizeTimers(currentTimersSettings))) {
		for (const timerSettings of newTimersSettings) {
			await properties.timerStorage.setShortTimer({
				type: timerSettings.type,
				timeOn: typeof timerSettings.timeOn === 'string' ? timerSettings.timeOn.replace(/^(\d{2}):(\d{2}).*/, '$1:$2:00') : null,
				timeOff: typeof timerSettings.timeOff === 'string' ? timerSettings.timeOff.replace(/^(\d{2}):(\d{2}).*/, '$1:$2:00') : null,
				weekdays: convertWeekday.fromLong(timerSettings.weekdays).toShort(),
			});
		}
		const timersToRemove = currentTimersSettings.filter(
			(currentTimer) => !newTimersSettings.some((newTimer) => currentTimer.type === newTimer.type),
		);
		for (const { type } of timersToRemove) {
			yield properties.timerStorage.deleteTimer(type);
		}
		return CheckAndSetResult.CHANGED;
	}
	return CheckAndSetResult.NOT_CHANGED;
}

export function sortByTimerTypeCallback(a: IShortWeekdayTimer | ILongWeekdayTimer, b: IShortWeekdayTimer | ILongWeekdayTimer) {
	if (a.type < b.type) {
		return -1;
	}
	if (a.type > b.type) {
		return 1;
	}
	return 0;
}

function normalizeTimers<T extends IShortWeekdayTimer | ILongWeekdayTimer>(timers: T[]) {
	let outputTimers = _.cloneDeep(timers);
	outputTimers.forEach((timer: T) => timer.weekdays.sort());
	outputTimers.sort(sortByTimerTypeCallback);
	return outputTimers;
}
