import { SetDeviceVolumeSucceeded } from '@signageos/actions/dist/Device/Audio/deviceAudioActions';
import { SetDeviceAutoRecoverySucceeded } from '@signageos/actions/dist/Device/AutoRecovery/deviceAutoRecoveryActions';
import { SetDeviceBrightnessSucceeded } from '@signageos/actions/dist/Device/Brightness/deviceBrightnessActions';
import { UpdateDeviceTimeSettingsSucceed } from '@signageos/actions/dist/Device/DateTime/deviceDateTimeActions';
import { SetDeviceDebugSettingsSucceeded } from '@signageos/actions/dist/Device/Debug/deviceDebugActions';
import { SetExtendedManagementUrlSucceeded } from '@signageos/actions/dist/Device/ExtendedManagement/deviceExtendedManagementActions';
import { SetDevicePeerRecoverySucceeded } from '@signageos/actions/dist/Device/PeerRecovery/devicePeerRecoveryActions';
import { ApplyPolicySucceeded } from '@signageos/actions/dist/Device/Policy/devicePolicyActions';
import {
	CancelScheduledPowerActionSucceed,
	PerformPowerActionSucceed,
	SchedulePowerActionSucceed,
} from '@signageos/actions/dist/Device/Power/devicePowerActions';
import { SetDeviceRemoteControlSettingsSucceeded } from '@signageos/actions/dist/Device/RemoteControl/deviceRemoteControlActions';
import { DeviceResizeSucceeded } from '@signageos/actions/dist/Device/Resolution/deviceResolutionActions';
import { UpdateDeviceTelemetryRecord } from '@signageos/actions/dist/Device/Telemetry/deviceTelemetryActions';
import { SetDeviceTimerSettingsSucceeded } from '@signageos/actions/dist/Device/Timer/deviceTimerActions';
import { MonitoringLogData } from '@signageos/common-types/dist/Device/MonitoringLog/MonitoringLogData';
import { DeviceSettingsType } from '@signageos/common-types/dist/Device/Settings/DeviceSettingsType';
import { DeviceTelemetryType } from '@signageos/common-types/dist/Device/Telemetry/DeviceTelemetryType';
import wait from '@signageos/lib/dist/Timer/wait';
import Debug from 'debug';
import _ from 'lodash';
import { takeEvery } from 'redux-saga/effects';
import { IntervalType, ITelemetryIntervals } from '../../../Device/Configuration/telemetryIntervals';
import { withDependencies } from '../../../DI/dependencyInjection';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import Responsibility from '../../../Feature/Responsibility';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import { PowerActionRules } from '../../../NativeDevice/Power/IPowerActionTimer';
import Property from '../../../Property/Property';
import { IPropertyStorage } from '../../../Property/propertyStorage';
import { IProprietaryTimerStorage } from '../../../Timer/ITimerStorage';
import { deliver } from './../../../Socket/socketActionDeliverHelper';
import { registerTelemetry } from './deviceTelemetryHelper';
import IGetTelemetryProperties from './IGetTelemetryProperties';
import { NetworkStatusChanged } from '../../../Front/Network/networkActions';
import { CryptographicKeyChanged } from '../Secrets/deviceCryptographicSaga';
import { ExecuteCustomScriptSucceeded } from '@signageos/actions/dist/Device/CustomScript/deviceCustomScriptActions';

const debug = Debug('@signageos/front-display:Management:Telemetry');

type IGetSettings<T extends DeviceTelemetryType> = (
	properties: IGetTelemetryProperties,
) => Promise<MonitoringLogData[T]> | Iterator<unknown>;

export const createDeviceTelemetrySaga = <T extends DeviceTelemetryType>(
	type: T,
	responsibility: Responsibility,
	requiredCapabilities: ManagementCapability[],
	getSettings: IGetSettings<T>,
) =>
	withDependencies(
		['telemetryIntervals'],
		function* (
			{
				telemetryIntervals,
			}: {
				telemetryIntervals: ITelemetryIntervals | null;
			},
			responsibilities: IResponsibilities,
			getManagementDriver: () => IManagementDriver,
			propertyStorage: IPropertyStorage,
			timerStorage: IProprietaryTimerStorage,
			powerActionRules: () => Promise<PowerActionRules>,
			applicationVersion: string,
		) {
			if (!responsibilities.has(Responsibility.TELEMETRY) || !responsibilities.has(responsibility)) {
				return;
			}

			for (const capability of requiredCapabilities) {
				const supported = yield getManagementDriver().managementSupports(capability);
				if (!supported) {
					return;
				}
			}

			yield takeEvery(ApplyPolicySucceeded, function* (actionType: ApplyPolicySucceeded<typeof type>) {
				if (actionType.name === type) {
					yield forceTelemetryUpdate(
						type,
						getSettings,
						getManagementDriver,
						propertyStorage,
						timerStorage,
						powerActionRules,
						applicationVersion,
					);
				}
			});

			yield takeEvery(ExecuteCustomScriptSucceeded, function* () {
				yield getSettingsAndSendItWhenChanged(
					type,
					getSettings,
					getManagementDriver,
					propertyStorage,
					timerStorage,
					powerActionRules,
					applicationVersion,
				);
			});

			const succeedAction = convertTypeToAction(type);
			if (succeedAction) {
				yield takeEvery(succeedAction, function* () {
					yield forceTelemetryUpdate(
						type,
						getSettings,
						getManagementDriver,
						propertyStorage,
						timerStorage,
						powerActionRules,
						applicationVersion,
					);
				});
				if (type === DeviceTelemetryType.POWER_ACTIONS_SCHEDULE) {
					yield takeEvery(CancelScheduledPowerActionSucceed, function* () {
						yield forceTelemetryUpdate(
							type,
							getSettings,
							getManagementDriver,
							propertyStorage,
							timerStorage,
							powerActionRules,
							applicationVersion,
						);
					});
				}
			}

			const telemetryIntervalType = convertTelemetryTypeToIntervalType(type);
			const periodMs = telemetryIntervals?.[telemetryIntervalType] ?? 0;
			yield registerTelemetry(type, periodMs, function* () {
				try {
					// Browser (localStorage) remembers the last telemetry data
					// When we switch app type (e.g. from CA to CC), there might be old saved telemetry data
					// so we need to force updated them, because the check for change will not pass
					if (type === DeviceTelemetryType.APP_MODULES) {
						yield* forceTelemetryUpdate(
							type,
							getSettings,
							getManagementDriver,
							propertyStorage,
							timerStorage,
							powerActionRules,
							applicationVersion,
						);
					}

					yield* getSettingsAndSendItWhenChanged(
						type,
						getSettings,
						getManagementDriver,
						propertyStorage,
						timerStorage,
						powerActionRules,
						applicationVersion,
					);
				} catch (error) {
					debug(`reporting device settings ${type} failed`, error);
				}
			});
		},
	);

function* getSettingsAndSendItWhenChanged<T extends DeviceTelemetryType>(
	type: T,
	getSettings: IGetSettings<T>,
	getManagementDriver: () => IManagementDriver,
	propertyStorage: IPropertyStorage,
	timerStorage: IProprietaryTimerStorage,
	powerActionRules: () => Promise<PowerActionRules>,
	applicationVersion: string,
) {
	const property: Property = getTelemetryPropertyName(type, 'LATEST_REPORTED_SETTINGS');
	const cachedDeviceSettings: MonitoringLogData[T] | undefined = yield propertyStorage.getValueOrDefault<MonitoringLogData[T] | undefined>(
		property,
		undefined,
	);
	const currentDeviceSettings: MonitoringLogData[T] = yield getSettings({
		managementDriver: getManagementDriver(),
		timerStorage,
		powerActionRules,
		propertyStorage,
		applicationVersion,
	});

	const isDeviceSettingsChanged = !_.isEqual(cachedDeviceSettings, currentDeviceSettings);
	if (isDeviceSettingsChanged) {
		yield deliver<UpdateDeviceTelemetryRecord<T, MonitoringLogData[T]>>({
			type: UpdateDeviceTelemetryRecord,
			name: type,
			data: currentDeviceSettings,
		});

		yield propertyStorage.setValue(property, currentDeviceSettings);
	}
}

function* forceTelemetryUpdate<T extends DeviceTelemetryType>(
	type: T,
	getSettings: IGetSettings<T>,
	getManagementDriver: () => IManagementDriver,
	propertyStorage: IPropertyStorage,
	timerStorage: IProprietaryTimerStorage,
	powerActionRules: () => Promise<PowerActionRules>,
	applicationVersion: string,
) {
	const property: Property = getTelemetryPropertyName(type, 'LATEST_REPORTED_SETTINGS');

	// Delay for resolving deviceLog
	yield wait(500);

	try {
		const currentDeviceSettings: MonitoringLogData[T] = yield getSettings({
			managementDriver: getManagementDriver(),
			timerStorage,
			powerActionRules,
			propertyStorage,
			applicationVersion,
		});

		yield deliver<UpdateDeviceTelemetryRecord<T, MonitoringLogData[T]>>({
			type: UpdateDeviceTelemetryRecord,
			name: type,
			data: currentDeviceSettings,
		});

		yield propertyStorage.setValue(property, currentDeviceSettings);
	} catch (error) {
		console.error('forceTelemetryUpdate failed', error);
	}
}

export function getPropertyName(type: DeviceSettingsType, prefix: string): Property {
	if (typeof DeviceSettingsType[type] === 'undefined') {
		throw Error(`Device settings type ${type} does not exist`);
	}
	const propertyKey = `${prefix}_${DeviceSettingsType[type]}`;
	if (typeof Property[propertyKey as keyof typeof Property] === 'undefined') {
		throw Error(`Property key is not valid ${propertyKey}`);
	}
	return Property[propertyKey as keyof typeof Property];
}
export function getTelemetryPropertyName(type: DeviceTelemetryType, prefix: string): Property {
	if (typeof DeviceTelemetryType[type] === 'undefined') {
		throw Error(`Device settings type ${type} does not exist`);
	}
	const propertyKey = `${prefix}_${DeviceTelemetryType[type]}`;
	if (typeof Property[propertyKey as keyof typeof Property] === 'undefined') {
		throw Error(`Property key is not valid ${propertyKey}`);
	}
	return Property[propertyKey as keyof typeof Property];
}

function convertTelemetryTypeToIntervalType<T extends DeviceTelemetryType>(type: T) {
	switch (type) {
		case DeviceTelemetryType.APPLICATION_VERSION:
			return IntervalType.APPLICATION_VERSION;
		case DeviceTelemetryType.BRIGHTNESS:
			return IntervalType.BRIGHTNESS;
		case DeviceTelemetryType.DATETIME:
			return IntervalType.DATETIME;
		case DeviceTelemetryType.DEBUG:
			return IntervalType.DEBUG;
		case DeviceTelemetryType.FIRMWARE_VERSION:
			return IntervalType.FIRMWARE_VERSION;
		case DeviceTelemetryType.ORIENTATION:
			return IntervalType.ORIENTATION;
		case DeviceTelemetryType.POWER_ACTIONS_SCHEDULE:
			return IntervalType.POWER_ACTIONS_SCHEDULE;
		case DeviceTelemetryType.PROPRIETARY_TIMERS:
			return IntervalType.PROPRIETARY_TIMERS;
		case DeviceTelemetryType.REMOTE_CONTROL:
			return IntervalType.REMOTE_CONTROL;
		case DeviceTelemetryType.RESOLUTION:
			return IntervalType.RESOLUTION;
		case DeviceTelemetryType.TEMPERATURE:
			return IntervalType.TEMPERATURE;
		case DeviceTelemetryType.TIMERS:
			return IntervalType.TIMERS;
		case DeviceTelemetryType.VOLUME:
			return IntervalType.VOLUME;
		case DeviceTelemetryType.BUNDLED_APPLET:
			return IntervalType.BUNDLED_APPLET;
		case DeviceTelemetryType.PROXY:
			return IntervalType.PROXY;
		case DeviceTelemetryType.WIFI_STRENGTH:
			return IntervalType.WIFI_STRENGTH;
		case DeviceTelemetryType.AUTO_RECOVERY:
			return IntervalType.AUTO_RECOVERY;
		case DeviceTelemetryType.PEER_RECOVERY:
			return IntervalType.PEER_RECOVERY;
		case DeviceTelemetryType.MANAGEMENT_CAPABILITIES:
			return IntervalType.MANAGEMENT_CAPABILITIES;
		case DeviceTelemetryType.NETWORK_INTERFACES:
			return IntervalType.NETWORK_INTERFACES;
		case DeviceTelemetryType.DISPLAY_POWER_ON:
			return IntervalType.DISPLAY_POWER_ON;
		case DeviceTelemetryType.APP_MODULES:
			return IntervalType.APP_MODULES;
		case DeviceTelemetryType.FEATURE_FLAGS:
			return IntervalType.FEATURE_FLAGS;
		case DeviceTelemetryType.CRYPTOGRAPHIC_KEY:
			return IntervalType.CRYPTOGRAPHIC_KEY;
		case DeviceTelemetryType.EXTENDED_MANAGEMENT:
			return IntervalType.EXTENDED_MANAGEMENT_URL;
		default:
			return IntervalType.DEFAULT;
	}
}

function convertTypeToAction<T extends DeviceTelemetryType>(type: T) {
	switch (type) {
		case DeviceTelemetryType.BRIGHTNESS:
			return SetDeviceBrightnessSucceeded;
		case DeviceTelemetryType.DATETIME:
			return UpdateDeviceTimeSettingsSucceed;
		case DeviceTelemetryType.DEBUG:
			return SetDeviceDebugSettingsSucceeded;
		case DeviceTelemetryType.ORIENTATION:
			return DeviceResizeSucceeded;
		case DeviceTelemetryType.POWER_ACTIONS_SCHEDULE:
			return SchedulePowerActionSucceed;
		case DeviceTelemetryType.PROPRIETARY_TIMERS:
			return SetDeviceTimerSettingsSucceeded;
		case DeviceTelemetryType.REMOTE_CONTROL:
			return SetDeviceRemoteControlSettingsSucceeded;
		case DeviceTelemetryType.RESOLUTION:
			return DeviceResizeSucceeded;
		case DeviceTelemetryType.TIMERS:
			return SetDeviceTimerSettingsSucceeded;
		case DeviceTelemetryType.VOLUME:
			return SetDeviceVolumeSucceeded;
		case DeviceTelemetryType.AUTO_RECOVERY:
			return SetDeviceAutoRecoverySucceeded;
		case DeviceTelemetryType.PEER_RECOVERY:
			return SetDevicePeerRecoverySucceeded;
		case DeviceTelemetryType.EXTENDED_MANAGEMENT:
			return SetExtendedManagementUrlSucceeded;
		case DeviceTelemetryType.NETWORK_INTERFACES:
			return NetworkStatusChanged;
		case DeviceTelemetryType.DISPLAY_POWER_ON:
			return PerformPowerActionSucceed;
		case DeviceTelemetryType.CRYPTOGRAPHIC_KEY:
			return CryptographicKeyChanged;
		default:
			return;
	}
}
