import { createDeviceTelemetrySaga } from '../Telemetry/deviceTelemetrySagas';
import { DeviceTelemetryType } from '@signageos/common-types/dist/Device/Telemetry/DeviceTelemetryType';
import Responsibility from '../../../Feature/Responsibility';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import IGetTelemetryProperties from '../Telemetry/IGetTelemetryProperties';

export const telemetryCryptographicSaga = createDeviceTelemetrySaga(
	DeviceTelemetryType.CRYPTOGRAPHIC_KEY,
	Responsibility.SECRET_MANAGEMENT,
	[ManagementCapability.SECRETS],
	getData,
);

export async function getData({ managementDriver }: IGetTelemetryProperties) {
	const keyData = await managementDriver.secretManager.getPublicKey();
	return {
		...algorithmToTelemetry(keyData.algorithm),
		...keyValidityToTelemetry(keyData.keyValidity),
		publicKey: {
			spki: keyData.publicKeySpki,
		},
	};
}

// https://developer.mozilla.org/en-US/docs/Web/API/SubtleCrypto/generateKey#parameters
const algorithmToTelemetry = (algorithm?: RsaHashedKeyAlgorithm) => {
	if (!algorithm) {
		return {};
	}

	const encryptionPaddings = [];
	switch (algorithm.name) {
		case 'RSA-OAEP':
			encryptionPaddings.push('OAEP');
			break;
		default:
			throw new Error(`Unsupported algorithm: ${algorithm.name}`);
	}

	return {
		algorithm: 'RSA',
		keySize: algorithm.modulusLength,
		digests: [algorithm.hash.name],
		encryptionPaddings,
	};
};

const keyValidityToTelemetry = (keyValidity?: { start?: Date; end?: Date }) => {
	if (keyValidity?.start || keyValidity?.end) {
		return {
			keyValidity: {
				start: keyValidity?.start,
				forOriginationEnd: keyValidity?.end,
				forConsumptionEnd: keyValidity?.end,
			},
		};
	} else {
		return {};
	}
};
