import INetwork, { INetworkOptions } from '../../../NativeDevice/Network/INetwork';
import { checksumString } from '../../../Hash/checksum';
import { getLocalIpAddresses } from '../../../NativeDevice/Network/ipAddressHelper';
import { NetworkInterfaces, NetworkInterfacesTypes } from '@signageos/common-types/dist/Device/Network/NetworkInterface';

export default class MockNetwork implements INetwork {
	private ifaces: NetworkInterfaces[] = [];

	constructor(private window: Window) {}

	public async listInterfaces(): Promise<NetworkInterfaces[]> {
		await this.ensureInterfaces();
		return this.ifaces;
	}

	public async setManual(ifaceName: string, options: INetworkOptions): Promise<void> {
		console.info('Emulated setManual for interface ' + ifaceName, options);
		this.setInterfaceDisabled(ifaceName, false);
	}

	public async setDHCP(ifaceName: string): Promise<void> {
		console.info('Emulated setDHCP for interface ' + ifaceName);
		this.setInterfaceDisabled(ifaceName, false);
	}

	public async disableInterface(ifaceName: string): Promise<void> {
		await this.ensureInterfaces();
		this.setInterfaceDisabled(ifaceName, true);
	}

	private setInterfaceDisabled(ifaceName: string, disabled: boolean) {
		for (const iface of this.ifaces) {
			if (iface.name === ifaceName) {
				iface.disabled = disabled;
			}
		}
	}

	private async ensureInterfaces() {
		if (this.ifaces.length === 0) {
			this.ifaces = await Promise.all([this.generateInterfaceInfo('ethernet'), this.generateInterfaceInfo('wifi')]);
		}
	}

	private async generateInterfaceInfo(type: NetworkInterfacesTypes): Promise<NetworkInterfaces> {
		const localAddress = await this.getIPAddress();
		switch (type) {
			case 'ethernet':
				return {
					type,
					name: 'eth0',
					localAddress,
					macAddress: this.generateMacAddress(20),
					gateway: this.getGateway(localAddress),
					netmask: '',
					dns: [],
				};
			case 'wifi':
				return {
					type,
					name: 'wlan0',
					localAddress,
					macAddress: this.generateMacAddress(15),
					gateway: this.getGateway(localAddress),
					netmask: '',
					dns: [],
					wifiStrength: 100,
					wifiSsid: 'SignageOS',
				};
			default:
				throw new Error('Unknown network interface type');
		}
	}

	private generateMacAddress(random: number) {
		return checksumString(checksumString(this.window.navigator.userAgent, random), 12)
			.match(/.{1,2}/g)!
			.join(':');
	}

	private async getIPAddress() {
		const localIpAddresses = await getLocalIpAddresses();
		return localIpAddresses.length > 0 ? localIpAddresses[0] : '192.168.1.113';
	}

	private getGateway(ipAddress: string) {
		const maskedIpAddress = ipAddress.split('.');
		maskedIpAddress.pop();
		return maskedIpAddress.join('.') + '.1'; // only fake gateway number
	}
}
