import { takeEveryAndBindWhenPlatform } from '../../../Socket/socketActionCreator';
import { PerformDeviceDeprovision } from '@signageos/actions/dist/Device/Deprovision/deviceDeprovisionActions';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import ICacheDriver from '../../../NativeDevice/ICacheDriver';
import wait from '@signageos/lib/dist/Timer/wait';
import { IStorageUnit, IFilePath } from '../../../NativeDevice/fileSystem';
import Responsibility from '../../../Feature/Responsibility';
import { IResponsibilities } from '../../../Feature/Responsibilities';
import { getObjectKeys } from '@signageos/lib/dist/Utils/object';
import NumberTimerType from '../../../NativeDevice/Timer/TimerType';
import { ManagementCacheDriver, normalizeManagementCacheDriver } from '../../../NativeDevice/Default/combinedDriver';

export function* deviceDeprovision(
	getNativeDriver: () => ManagementCacheDriver,
	responsibilities: IResponsibilities,
	sessionIdKey: string,
) {
	if (responsibilities.has(Responsibility.DEPROVISIONING)) {
		yield takeEveryAndBindWhenPlatform(PerformDeviceDeprovision, function* () {
			try {
				let { managementDriver } = normalizeManagementCacheDriver(getNativeDriver());
				yield Promise.allSettled([
					clearCache(() => normalizeManagementCacheDriver(getNativeDriver()).cacheDriver),
					clearFileSystem(() => normalizeManagementCacheDriver(getNativeDriver()).managementDriver),
					managementDriver.remoteControlSetEnabled(true),
					clearTimers(() => normalizeManagementCacheDriver(getNativeDriver()).managementDriver),
					managementDriver.setSessionId(sessionIdKey, ''),
					managementDriver.screenSetBrightness('00:00', 50, '00:00', 50),
					managementDriver.setVolume(50),
					managementDriver.displayPowerOn(),
					managementDriver.setDebug(false),
				]);
				yield wait(60e3); // wait until front saga is 100% also done
				yield normalizeManagementCacheDriver(getNativeDriver()).managementDriver.systemReboot();
			} catch (error) {
				console.error('deviceDeprovision failed', error);
			}
		});
	}
}

async function clearCache(getNativeDriver: () => ICacheDriver) {
	const uids = await getNativeDriver().cacheGetUids();
	await Promise.allSettled(
		uids.map(async (uid: string) => {
			await getNativeDriver().cacheDelete(uid);
		}),
	);
}

async function clearFileSystem(getNativeDriver: () => IManagementDriver) {
	const storageUnits = await getNativeDriver().fileSystem.listStorageUnits();
	const internalStorageUnits = storageUnits.filter((su: IStorageUnit) => !su.removable);
	await Promise.allSettled(
		internalStorageUnits.map(async (storageUnit: IStorageUnit) => {
			const filePaths = await getNativeDriver().fileSystem.listFiles({ storageUnit, filePath: '' });
			await Promise.allSettled(
				filePaths.map(async (filePath: IFilePath) => {
					await getNativeDriver().fileSystem.deleteFile(filePath, true);
				}),
			);
		}),
	);
}

async function clearTimers(getNativeDriver: () => IManagementDriver) {
	await Promise.allSettled(
		getObjectKeys(NumberTimerType).map(async (timerTypeString) => {
			await getNativeDriver().setTimer(NumberTimerType[timerTypeString], null, null, [], 0);
		}),
	);
}
