import { fork, apply } from 'redux-saga/effects';
import wait from '@signageos/lib/dist/Timer/wait';
import { now as getCurrentDate } from '@signageos/lib/dist/DateTime/dateTimeFactory';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import { resolveCurrentBrightness } from '../../../NativeDevice/Screen/screenHelper';
import Property from '../../../Property/Property';
import { IPropertyStorage } from '../../../Property/propertyStorage';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import IBrightness from '../../../NativeDevice/IBrightness';

export function* brightnessRefresh(getNativeDriver: () => IManagementDriver, propertyStorage: IPropertyStorage) {
	// only start this process if the device doesn't support brightness schedule natively
	const isSupported: boolean = yield getNativeDriver().managementSupports(ManagementCapability.BRIGHTNESS_SCHEDULING);
	if (!isSupported) {
		yield fork(function* () {
			while (true) {
				yield wait(30 * 1e3);
				try {
					const now = getCurrentDate().toDate();
					yield refreshBrightness(getNativeDriver(), propertyStorage, now);
				} catch (error) {
					console.error('brightness refresh failed', error);
				}
			}
		});
	}
}

function* refreshBrightness(nativeDriver: IManagementDriver, propertyStorage: IPropertyStorage, now: Date) {
	const settings: IBrightness | null = yield propertyStorage.getValueOrDefault<IBrightness | null>(Property.BRIGHTNESS_SETTINGS, null);

	if (settings) {
		const newBrightness = resolveCurrentBrightness(settings.timeFrom1, settings.brightness1, settings.timeFrom2, settings.brightness2, now);

		const oldBrightnessSettings: IBrightness = yield nativeDriver.screenGetBrightness();
		const oldBrightness = resolveCurrentBrightness(
			oldBrightnessSettings.timeFrom1,
			oldBrightnessSettings.brightness1,
			oldBrightnessSettings.timeFrom2,
			oldBrightnessSettings.brightness2,
			now,
		);

		if (oldBrightness !== newBrightness) {
			yield apply(nativeDriver, nativeDriver.screenSetBrightness, [settings.timeFrom1, newBrightness, settings.timeFrom2, newBrightness]);
		}
	}
}
