import Raven from 'raven-js';

export function useRavenLogging(window: Window) {
	// override default console logging
	const consoleError = console.error;
	console.error = (...args: any[]) => {
		consoleError.call(console, ...args);
		captureRaven('error', 'error', ...args);
	};
	(window as any).onunhandledrejection = ({ promise, reason }: { promise: Promise<any>; reason: Error }) => console.error(reason, promise);
}

type LogLevel = 'critical' | 'error' | 'warn' | 'info' | 'debug';

function captureRaven(level: LogLevel, source: string, ...args: any[]) {
	const options = {
		level,
		tags: {
			source,
		},
		extra: null as any,
	};
	const exception = args.find((arg: any) => arg instanceof Error);
	if (exception) {
		options.extra = args.filter((arg: any) => arg !== exception);
		Raven.captureException(exception, options);
	} else {
		let message = args[0];
		if (typeof message !== 'string') {
			message = JSON.stringify(message);
		}
		Raven.captureMessage(message, options);
	}
}
