import * as crypto from 'crypto';
import * as crc32 from 'crc-32';
import type { Blob } from 'buffer';

export function checksumString(message: string, length: number = 50) {
	const hash = crypto.createHash('sha256');
	hash.update(message);
	return hash.digest('hex').substring(0, length);
}

export async function getChecksumOfFile(window: Window, uri: string, algorithm: string) {
	const response = await window.fetch(uri);
	return await getChecksumOfBlob(await response.blob(), algorithm);
}

export async function getChecksumOfBlob(blob: Blob | globalThis.Blob, algorithm: string) {
	const data = Buffer.from(await blob.arrayBuffer());
	if (algorithm === 'md5') {
		return crypto.createHash('md5').update(data).digest('hex');
	} else if (algorithm === 'crc32') {
		const hashNumber = crc32.buf(data);
		const hashUint32 = hashNumber < 0 ? hashNumber + Math.pow(16, 8) : hashNumber;
		return hashUint32.toString(16).padStart(8, '0');
	}
	throw new Error(`Checksum algorithm is not supported: ${algorithm}`);
}
