import { EventEmitter } from 'events';
import { cloneDeep } from 'lodash';
import './front.sass';
import * as React from 'react';
import * as ReactDOM from 'react-dom';
import { Store } from 'redux';
import Main from '../Front/Web/Display/Main';
import { createApplication } from '../Application/applicationFactory';
import { frontReducer, IFrontState } from './frontReducers';
import { frontSaga } from './frontSagas';
import createPropertyStorage from '../Property/propertyStorage';
import Property from '../Property/Property';
import { IWeinreContainer, startWeinreDebug } from './Device/Debug/deviceDebugHelper';
import ISynchronizer from '../Synchronization/ISynchronizer';
import { IWebWorkerFactory } from '../WebWorker/masterWebWorkerFactory';
import { IFrontManagementDriver } from '../NativeDevice/Management/IManagementDriver';
import { SubscriptionType } from '../Display/IConfig';
import Responsibility from '../Feature/Responsibility';
import { IBundledApplet } from './Applet/BundledApplet/bundledAppletActions';
import { createAbsentResponsibilities } from '../Feature/Responsibilities';
import { FrontCacheDriver, normalizeFrontCacheDriver } from '../NativeDevice/Default/combinedDriver';

export type IGlobalContainer = Window &
	IWeinreContainer & {
		WeinreServerURL?: string;
		WeinreServerId?: string;
		document: Window['document'];
		__front?: IFrontOptions & {
			store: Store<IFrontState>;
			storeEmitter: EventEmitter;
			onUnload: () => void;
			lastSagaError?: Error;
		};
	};

export interface IFrontOptions {
	global: IGlobalContainer;
	baseUrl: string;
	platformUri: string;
	staticBaseUrl: string;
	uploadBaseUrl: string;
	checkInterval: number;
	publicKey: string;
	sessionIdKey: string;
	frontAppletPrefix: string;
	frontDisplayVersion: string;
	weinreServerUrl: string;
	/** @deprecated use UpdateExtendedManagementUrl action instead */
	extendedManagementUrl: string | null;
	nativeDriver: FrontCacheDriver;
	managementDriver: IFrontManagementDriver;
	synchronizer: ISynchronizer;
	webWorkerFactory: IWebWorkerFactory;
	applicationVersion: string;
	subscriptionType: SubscriptionType;
	bundledApplet: null | IBundledApplet;
	autoVerification: { organizationUid: string; deviceName?: string } | undefined;
	excludedResponsibilities: Responsibility[];
	/** Enable cutting url of applet and offline cache files to shorter form so SSSP2 will be able to handle with them */
	shortAppletFilesUrl?: boolean;
}

export default async (options: IFrontOptions) => {
	const { frontDriver, cacheDriver } = normalizeFrontCacheDriver(options.nativeDriver);

	const propertyStorage = createPropertyStorage(() => cacheDriver);

	try {
		const weinreServerUrl = await propertyStorage.getValueOrDefault<string>(Property.DEBUG_WEINRE_SERVER_URL, options.weinreServerUrl);
		options.global.WeinreServerURL = weinreServerUrl;
		options.global.WeinreServerId = await frontDriver.getDeviceUid();
		const weinreDebugEnabled = await propertyStorage.getValueOrDefault(Property.DEBUG_WEINRE_ENABLED, false);
		if (weinreDebugEnabled) {
			startWeinreDebug(options.global);
		}
	} catch (error) {
		console.error(error);
	}

	const subscriptionType = options.subscriptionType;
	const { excludedResponsibilities: _, ...newOptions } = options;
	const {
		baseUrl,
		platformUri,
		staticBaseUrl,
		uploadBaseUrl,
		checkInterval,
		weinreServerUrl: weinreUri,
		extendedManagementUrl,
		...sagaOptions
	} = {
		responsibilities: createAbsentResponsibilities(...options.excludedResponsibilities),
		...newOptions,
	};
	const config = {
		baseUrl,
		platformUri,
		staticBaseUrl,
		uploadBaseUrl,
		weinreUri,
		extendedManagementUrl,
		subscriptionType,
		checkInterval,
	};

	let lastSagaError: Error | undefined;

	const application = await createApplication(
		frontReducer,
		(store: Store<IFrontState>) => frontSaga(config, sagaOptions, store.getState, propertyStorage, store.dispatch),
		() => frontDriver,
		(e: Error) => {
			lastSagaError = cloneDeep(e);
			console.warn('Sagas broken', e);
		},
	);

	ReactDOM.render(<Main store={application.store} />, options.global.document.getElementById('body'));

	options.global.__front = {
		...options,
		store: application.store,
		storeEmitter: application.storeEmitter,
		onUnload: application.onUnload,
		lastSagaError,
	};
};
