import * as React from 'react';
import { connect } from 'react-redux';
import { IFrontState } from '../../../frontReducers';
import MainInfo from '../../Info/MainInfo';
import { Action } from 'redux';
import InfoProperty from '../../Info/InfoProperty';
import DeviceInfo from '../../Info/DeviceInfo';
import HashtagIcon from '../../Application/Icons/HashtagIcon';
import OfflineIcon from '../../Application/Icons/OfflineIcon';
import styles from './DeviceVerification.module.sass';
import TimeInfo from '../../Info/TimeInfo';
import ConnectingIcon from '../../Application/Icons/ConnectingIcon';
import DangerIcon from '../../Application/Icons/DangerIcon';

export interface IStateProps {
	children: React.ReactNode[];
	verificationHash: string;
	verified: boolean;
	model: string | null;
	ipAddress: string | null;
	isSerialNumberSupported: boolean;
	serialNumber: string | null;
	pinCode: string | null;
	network: string;
	applicationType: string | null;
	isOnline: boolean;
	isConnected: boolean;
	applicationVersion: string;
	timestamp: number | null;
	timezone: string | null;
	firmwareVersion: string;
	osVersion: string;
	frontDisplayVersion: string;
}
export interface IDispatchProps {}

export class DeviceVerification extends React.PureComponent<IStateProps & IDispatchProps> {
	public render() {
		if (!this.props.verified) {
			const deviceInfo = (
				<div>
					<TimeInfo timestamp={this.props.timestamp} timezone={this.props.timezone} />
					<DeviceInfo>
						{/* Device specific */}
						<InfoProperty title='Device:'>{this.getDeviceModel()}</InfoProperty>
						<InfoProperty title='System:'>{this.getSystemLabel()}</InfoProperty>
						<InfoProperty title='System Version:'>{this.getOsVersion()}</InfoProperty>
						<InfoProperty title='Firmware Version:'>{this.getFirmwareVersion()}</InfoProperty>
						<InfoProperty title='Serial Number:'>{this.getDeviceSerialNumber()}</InfoProperty>

						{/* Application specific */}
						{this.props.frontDisplayVersion ? (
							<InfoProperty title='Front Display Version:'>{this.props.frontDisplayVersion}</InfoProperty>
						) : null}
						{this.props.applicationVersion ? (
							<InfoProperty title='Application Version:'>{this.props.applicationVersion}</InfoProperty>
						) : null}

						{/* Network specific */}
						{this.props.network ? <InfoProperty title='Network:'>{this.props.network}</InfoProperty> : null}
						{this.props.ipAddress ? <InfoProperty title='Local IP Address:'>{this.props.ipAddress}</InfoProperty> : null}

						{/* Other specifics */}
						<InfoProperty title='PIN:'>{this.getPinCode()}</InfoProperty>
						<InfoProperty title='Setup menu:'>PIN + OK/Enter opens OSD</InfoProperty>
						<InfoProperty title='Status:'>Device is not verified</InfoProperty>
					</DeviceInfo>
				</div>
			);
			if (!this.props.isOnline) {
				return (
					<div key={'offline'} className={styles.wrapper}>
						<MainInfo title='No internet connection' value='' />
						{deviceInfo}
						<OfflineIcon />
					</div>
				);
			} else if (!this.props.isConnected) {
				return (
					<div key={'serverDisconnected'} className={styles.wrapper}>
						<MainInfo title='Connecting to the server...' value='' />
						{deviceInfo}
						<ConnectingIcon />
					</div>
				);
			} else if (this.props.verificationHash) {
				return (
					<div key={'verificationHash'} className={styles.wrapper}>
						<MainInfo title='Verification Hash:' value={this.props.verificationHash} />
						{deviceInfo}
						<HashtagIcon />
					</div>
				);
			} else if (this.props.isSerialNumberSupported && !this.props.serialNumber) {
				return (
					<div key={'noSerialNumber'} className={styles.wrapper}>
						<MainInfo
							title='Device registration:'
							value='Unable to register -
						No serial number'
						/>
						{deviceInfo}
						<DangerIcon />
					</div>
				);
			} else {
				return (
					<div key={'inProgress'} className={styles.wrapper}>
						<MainInfo title='Device registration:' value={'in progress'} />
						{deviceInfo}
						<HashtagIcon />
					</div>
				);
			}
		} else {
			return <div>{this.props.children}</div>;
		}
	}

	private getSystemLabel() {
		return this.props.applicationType ? this.props.applicationType.toUpperCase() : 'UNKNOWN';
	}

	private getDeviceModel() {
		return this.props.model ? this.props.model : 'Unknown';
	}

	private getDeviceSerialNumber() {
		return this.props.serialNumber ? this.props.serialNumber : 'Unknown';
	}

	private getPinCode() {
		return this.props.pinCode ? this.props.pinCode : '####';
	}

	private getOsVersion() {
		return this.props.osVersion ? this.props.osVersion : 'Unknown';
	}

	private getFirmwareVersion() {
		return this.props.firmwareVersion ? this.props.firmwareVersion : 'Unknown';
	}
}
export default connect(
	(state: IFrontState) => {
		return {
			verificationHash: state.deviceVerification.verificationHash,
			verified: state.deviceVerification.verified,
			applicationType: state.device.applicationType,
			model: state.device.model,
			ipAddress: state.device.ipAddress,
			isSerialNumberSupported: state.device.isSerialNumberSupported,
			serialNumber: state.device.serialNumber,
			pinCode: state.device.pinCode,
			network: state.device.activeInterface,
			isOnline: state.network.connected,
			isConnected: state.socket.connected,
			applicationVersion: state.device.applicationVersion,
			timestamp: state.device.timestamp,
			timezone: state.device.timezone,
			firmwareVersion: state.device.firmwareVersion,
			osVersion: state.device.osVersion,
			frontDisplayVersion: state.device.frontDisplayVersion,
		} as IStateProps;
	},
	(_dispatch: (action: Action) => void) => ({}),
)(DeviceVerification);
