import * as React from 'react';
import { IFile as IFileSystemFile } from '../../../NativeDevice/fileSystem';

import styles from './ActiveApplet.module.sass';

export interface IProps {
	appletUid: string;
	checksum: string;
	uniqueHash?: string;
	binaryFile?: IFileSystemFile;
	nextAppletUid?: string;
	nextChecksum?: string;
	nextBinaryFile?: IFileSystemFile;
	onCreated(iframeElement: HTMLIFrameElement, appletUid: string): void;
	onStart(iframeElement: HTMLIFrameElement, appletUid: string): void;
	onRemoved(id: string): void;
}

export default class GaplessApplet extends React.PureComponent<IProps, {}> {
	private wrapper: HTMLDivElement;
	private currentIframe: null | {
		checksum: string;
		uniqueHash: string;
		element: HTMLIFrameElement;
	} = null;
	private nextIframe: null | {
		checksum: string;
		element: HTMLIFrameElement;
	} = null;

	public componentDidMount() {
		this.updateIframes();
	}

	public componentDidUpdate() {
		this.updateIframes();
	}

	public render() {
		return <div ref={(element: HTMLDivElement) => (this.wrapper = element)}></div>;
	}

	private updateIframes() {
		if (!this.props.uniqueHash || !this.props.binaryFile) {
			return;
		}

		if (!this.currentIframe) {
			this.createCurrentIframe(this.props.appletUid, this.props.checksum, this.props.uniqueHash, this.props.binaryFile);
		} else if (this.currentIframe.checksum === this.props.checksum && this.currentIframe.uniqueHash !== this.props.uniqueHash) {
			this.createCurrentIframe(this.props.appletUid, this.props.checksum, this.props.uniqueHash, this.props.binaryFile);
		} else if (this.currentIframe.checksum !== this.props.checksum) {
			if (this.nextIframe && this.props.checksum === this.nextIframe.checksum) {
				this.makeNextIframeCurrent(this.props.appletUid, this.props.uniqueHash);
			} else {
				this.createCurrentIframe(this.props.appletUid, this.props.checksum, this.props.uniqueHash, this.props.binaryFile);
			}
		}

		if (
			this.props.nextAppletUid &&
			this.props.nextChecksum &&
			this.props.nextBinaryFile &&
			this.currentIframe &&
			this.currentIframe.checksum !== this.props.nextChecksum
		) {
			this.createNextIframe(this.props.nextAppletUid, this.props.nextChecksum, this.props.nextBinaryFile);
		}
	}

	private createCurrentIframe(appletUid: string, checksum: string, uniqueHash: string, binaryFile: IFileSystemFile) {
		if (this.currentIframe) {
			const removedId = this.currentIframe.element.id;
			this.wrapper.removeChild(this.currentIframe.element);
			this.props.onRemoved(removedId);
		}

		const iframe = document.createElement('iframe');
		iframe.id = 'applet-' + checksum;
		iframe.src = binaryFile.localUri;
		iframe.setAttribute('scrolling', 'no');
		iframe.setAttribute('allow', 'geolocation *; microphone *; camera *; midi *; encrypted-media *; autoplay *;');
		iframe.setAttribute('class', styles.appletIframe);
		this.wrapper.appendChild(iframe);
		this.currentIframe = { checksum, uniqueHash, element: iframe };
		this.props.onCreated(iframe, appletUid);
		this.props.onStart(iframe, appletUid);
		iframe.focus();
	}

	private createNextIframe(appletUid: string, checksum: string, binaryFile: IFileSystemFile) {
		if (this.nextIframe) {
			const removedId = this.nextIframe.element.id;
			this.wrapper.removeChild(this.nextIframe.element);
			this.props.onRemoved(removedId);
		}

		const iframe = document.createElement('iframe');
		iframe.id = 'applet-' + checksum;
		iframe.src = binaryFile.localUri;
		iframe.style.display = 'none';
		iframe.setAttribute('scrolling', 'no');
		iframe.setAttribute('allow', 'geolocation *; microphone *; camera *; midi *; encrypted-media *; autoplay *;');
		iframe.setAttribute('class', styles.appletIframe);
		this.wrapper.appendChild(iframe);
		this.nextIframe = { checksum, element: iframe };
		this.props.onCreated(iframe, appletUid);
	}

	private makeNextIframeCurrent(appletUid: string, uniqueHash: string) {
		if (this.nextIframe) {
			const iframe = this.nextIframe.element;
			iframe.style.display = 'block';
			if (this.currentIframe) {
				const removedId = this.currentIframe.element.id;
				this.wrapper.removeChild(this.currentIframe.element);
				this.props.onRemoved(removedId);
			}
			this.currentIframe = {
				checksum: this.nextIframe.checksum,
				uniqueHash,
				element: iframe,
			};
			this.nextIframe = null;
			this.props.onStart(iframe, appletUid);
			iframe.focus();
		}
	}
}
