import * as React from 'react';
import { connect } from 'react-redux';
import { IFrontState } from '../../frontReducers';
import { IFile as IFileSystemFile } from '../../../NativeDevice/fileSystem';
import MainInfo from '../Info/MainInfo';
import InfoProperty from '../Info/InfoProperty';
import GaplessApplet from './GaplessApplet';
import { CreateAppletIframe, RemoveAppletIframe, StartAppletIframe } from '../../AppletTiming/appletTimingActions';
import DeviceInfo from '../Info/DeviceInfo';
import SuccessIcon from '../Application/Icons/SuccessIcon';
import 'typeface-muli';
import TimeInfo from '../Info/TimeInfo';

export interface IStateProps {
	appletUid: string;
	appletBinaryFile: IFileSystemFile;
	appletMissing: boolean;
	appletUniqueHash: string;
	appletChecksum: string;
	nextAppletUid?: string;
	nextAppletBinaryFile?: IFileSystemFile;
	nextAppletChecksum?: string;
	deviceName: string;
	model: string | null;
	organizationTitle: string;
	appletIsMuted: boolean;
	applicationType: string | null;
	ethernetMacAddress: string | null;
	wifiMacAddress: string | null;
	serialNumber: string | null;
	frontDisplayVersion: string;
	applicationVersion: string;
	timezone: string;
	timestamp: number;
	ipAddress: string;
	interfaceName: string;
	osVersion: string;
	firmwareVersion: string;
}
export interface IDispatchProps {
	onAppletIframeCreated(appletUid: string, appletIframeId: string): void;
	onAppletIframeStarted(appletUid: string, appletIframeId: string): void;
	onAppletIframeRemoved(appletIframeId: string): void;
}

export class ActiveApplet extends React.PureComponent<IStateProps & IDispatchProps> {
	public render() {
		if (this.props.appletIsMuted) {
			return (
				<div>
					<MainInfo title='Device name:' value={this.props.deviceName || '-- No Name --'}>
						Content is muted because display is in POWER OFF state by timer settings
					</MainInfo>
				</div>
			);
		} else if (this.props.appletMissing) {
			return (
				<div>
					<MainInfo title='Device name:' value={this.props.deviceName || '-- No Name --'} />
					<TimeInfo timestamp={this.props.timestamp} timezone={this.props.timezone} />
					<DeviceInfo>
						{/* Device specific */}
						<InfoProperty title='Device:'>{this.getDeviceModel()}</InfoProperty>
						<InfoProperty title='System:'>{this.getSystemLabel()}</InfoProperty>
						{this.props.osVersion ? <InfoProperty title='System Version:'>{this.props.osVersion}</InfoProperty> : null}
						{this.props.firmwareVersion ? <InfoProperty title='Firmware Version:'>{this.props.firmwareVersion}</InfoProperty> : null}
						{this.props.serialNumber ? <InfoProperty title='Serial Number:'>{this.props.serialNumber}</InfoProperty> : null}

						{/* Application specific */}
						<InfoProperty title='Front Display Version:'>{this.props.frontDisplayVersion}</InfoProperty>
						<InfoProperty title='Application Version:'>{this.props.applicationVersion}</InfoProperty>
						{this.props.organizationTitle ? <InfoProperty title='Organization:'>{this.props.organizationTitle}</InfoProperty> : null}

						{/* Network specific */}
						{this.props.interfaceName ? <InfoProperty title='Interface:'>{this.props.interfaceName}</InfoProperty> : null}
						{this.props.ipAddress ? <InfoProperty title='Local IP Address:'>{this.props.ipAddress}</InfoProperty> : null}
						{this.props.wifiMacAddress ? <InfoProperty title='Wifi Mac Address:'>{this.props.wifiMacAddress}</InfoProperty> : null}
						{this.props.ethernetMacAddress ? (
							<InfoProperty title='Ethernet Mac Address:'>{this.props.ethernetMacAddress}</InfoProperty>
						) : null}

						{/* Other specifics */}
						<InfoProperty title='Status:'>No content applet configured</InfoProperty>
					</DeviceInfo>
					<SuccessIcon />
				</div>
			);
		} else if (this.props.appletChecksum) {
			return (
				<GaplessApplet
					appletUid={this.props.appletUid}
					checksum={this.props.appletChecksum}
					uniqueHash={this.props.appletUniqueHash}
					binaryFile={this.props.appletBinaryFile}
					nextAppletUid={this.props.nextAppletUid}
					nextChecksum={this.props.nextAppletChecksum}
					nextBinaryFile={this.props.nextAppletBinaryFile}
					onCreated={(iframeElement: HTMLIFrameElement, appletUid: string) => this.props.onAppletIframeCreated(appletUid, iframeElement.id)}
					onStart={(iframeElement: HTMLIFrameElement, appletUid: string) => this.props.onAppletIframeStarted(appletUid, iframeElement.id)}
					onRemoved={(iframeId: string) => this.props.onAppletIframeRemoved(iframeId)}
				/>
			);
		} else {
			return null;
		}
	}

	private getSystemLabel() {
		return this.props.applicationType ? this.props.applicationType.toUpperCase() : 'UNKNOWN';
	}

	private getDeviceModel() {
		return this.props.model ? this.props.model : 'Unknown';
	}
}

export default connect(
	(state: IFrontState) =>
		({
			appletUid: state.applet.activeAppletUid,
			appletBinaryFile: state.applet.activeAppletBinaryFile,
			appletMissing: state.applet.activeAppletMissing,
			appletUniqueHash: state.applet.activeAppletUniqueHash,
			appletChecksum: state.applet.activeTimingChecksum,
			nextAppletUid: state.applet.nextApplet ? state.applet.nextApplet.uid : undefined,
			nextAppletBinaryFile: state.applet.nextApplet ? state.applet.nextApplet.appletBinaryFile : undefined,
			nextAppletChecksum: state.applet.nextApplet ? state.applet.nextApplet.timingChecksum : undefined,
			deviceName: state.device.deviceName,
			model: state.device.model,
			organizationTitle: state.organization.organizationTitle,
			appletIsMuted: state.applet.appletIsMuted,
			applicationType: state.device.applicationType,
			ethernetMacAddress: state.device.ethernetMacAddress,
			wifiMacAddress: state.device.wifiMacAddress,
			serialNumber: state.device.serialNumber,
			frontDisplayVersion: state.device.frontDisplayVersion,
			timezone: state.device.timezone,
			timestamp: state.device.timestamp,
			applicationVersion: state.device.applicationVersion,
			ipAddress: state.device.ipAddress,
			interfaceName: state.device.activeInterface,
			osVersion: state.device.osVersion,
			firmwareVersion: state.device.firmwareVersion,
		}) as IStateProps,
	(dispatch: (action: CreateAppletIframe | StartAppletIframe | RemoveAppletIframe) => void) =>
		({
			onAppletIframeCreated: (appletUid: string, appletIframeId: string) =>
				dispatch({
					type: CreateAppletIframe,
					uid: appletUid,
					id: appletIframeId,
				} as CreateAppletIframe),
			onAppletIframeStarted: (appletUid: string, appletIframeId: string) =>
				dispatch({
					type: StartAppletIframe,
					uid: appletUid,
					id: appletIframeId,
				} as StartAppletIframe),
			onAppletIframeRemoved: (appletIframeId: string) =>
				dispatch({
					type: RemoveAppletIframe,
					id: appletIframeId,
				} as RemoveAppletIframe),
		}) as IDispatchProps,
)(ActiveApplet);
