import IFrontDriver from '../../NativeDevice/Front/IFrontDriver';
import { IStorageUnit } from '../../NativeDevice/fileSystem';
import FrontCapability from '../../NativeDevice/Front/FrontCapability';
import * as t from './../../Test/TestFramework';
import { TestCase } from '@signageos/actions/dist/Device/Test/deviceTestActions';

const TEST_DIRECTORY_PATH = 'test';
const host = 'https://2.signageos.io/assets';

const VIDEO1_URI = `${host}/test-videos-03_AME/video-test-03_15s_1920x1080_2fe7b039750a134aeac1c0a515710007.mp4`;
const VIDEO2_URI = `${host}/test-videos-04_AME/video-test-04_15s_1920x1080_88f90af5fa281d7efb8eae17848e71d9.mp4`;
const VIDEO_4K_URI = `${host}/test-videos-02_AME/video-test-02_15s_3840x2160_09c01fa898c80160edde2a1218ac6043.mp4`;
const VIDEO_SMALL_URI = `${host}/test-videos-03_AME/video-test-03_15s_320x200_5ce761cc9739c9a2d4c55be9ca7a8763.mp4`;

const VIDEO1_FILENAME = 'video1.mp4';
const VIDEO2_FILENAME = 'video2.mp4';
const VIDEO_4K_FILENAME = 'video_4k.mp4';
const VIDEO_SMALL_FILENAME = 'video_small.mp4';

let nativeDriver: IFrontDriver;

export default async (loadNativeDriver: IFrontDriver) => {
	nativeDriver = loadNativeDriver;
	return t.describe(TestCase.VIDEO, function* () {
		yield t.before('download video files', async () => {
			const internalStorageUnit = await getInternalStorageUnit();
			const testDirectoryFilePath = { storageUnit: internalStorageUnit, filePath: TEST_DIRECTORY_PATH };
			if (await nativeDriver.fileSystem.exists(testDirectoryFilePath)) {
				await nativeDriver.fileSystem.deleteFile(testDirectoryFilePath, true);
			}
			await nativeDriver.fileSystem.createDirectory(testDirectoryFilePath);
			await Promise.all([
				downloadVideoFile(VIDEO1_FILENAME, VIDEO1_URI),
				downloadVideoFile(VIDEO2_FILENAME, VIDEO2_URI),
				downloadVideoFile(VIDEO_4K_FILENAME, VIDEO_4K_URI),
				downloadVideoFile(VIDEO_SMALL_FILENAME, VIDEO_SMALL_URI),
			]);
		});
		yield t.after('delete test files', async () => {
			const internalStorageUnit = await getInternalStorageUnit();
			const testDirectoryFilePath = { storageUnit: internalStorageUnit, filePath: TEST_DIRECTORY_PATH };
			await nativeDriver.fileSystem.deleteFile(testDirectoryFilePath, true);
		});
		yield t.afterEach('clear nativeDriver.video', async () => {
			await nativeDriver.video.clearAll();
		});
		// ? beforeEach (await nativeDriver.video.clearAll();)?

		yield t.it('testPrepareVideo', async () => {
			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO1_FILENAME);
			await nativeDriver.video.prepare(videoFile.localUri, 0, 0, 1920, 1080);
		});
		yield t.it('testPrepareMultipleVideos', async () => {
			const videoMaxCount = nativeDriver.video.getMaxVideoCount();
			if (videoMaxCount === 1) {
				t.skip('Only 1 video is supported');
				return;
			}
			if (videoMaxCount === null) {
				t.skip('The max allowed videos is not specified');
				return;
			}
			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO_SMALL_FILENAME);
			for (let i = 0; i < videoMaxCount; i++) {
				await nativeDriver.video.prepare(videoFile.localUri, i * 100, 0, 100, 100);
			}
		});
		yield t.it('testPrepareVideoFailsWhenAllVideosArePlaying', async () => {
			const videoMaxCount = nativeDriver.video.getMaxVideoCount();
			if (videoMaxCount === null) {
				t.skip('Max video count is not defined');
				return;
			}

			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO_SMALL_FILENAME);
			for (let i = 0; i < videoMaxCount; i++) {
				await nativeDriver.video.play(videoFile.localUri, i * 100, 0, 100, 100);
			}

			await nativeDriver.video.prepare(videoFile.localUri, videoMaxCount * 100, 0, 100, 100).then(
				() => {
					t.skip("prepare didn't fail");
				},
				(_expected: unknown) => {
					// OK
				},
			);
		});
		yield t.it('testPlayVideo', async () => {
			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO1_FILENAME);
			const videoEventEmitter = await nativeDriver.video.play(videoFile.localUri, 0, 0, 1920, 1080);
			await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
				videoEventEmitter.on('ended', resolve);
				videoEventEmitter.on('error', () => reject(new Error('test video play failed')));
			});
		});
		yield t.it('testPrepareAndPlayVideo', async () => {
			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO1_FILENAME);
			await nativeDriver.video.prepare(videoFile.localUri, 0, 0, 1920, 1080);
			const videoEventEmitter = await nativeDriver.video.play(videoFile.localUri, 0, 0, 1920, 1080);
			await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
				videoEventEmitter.on('ended', resolve);
				videoEventEmitter.on('error', () => reject(new Error('test video play failed')));
			});
		});
		yield t.it('testPlay2Videos', async () => {
			const videoMaxCount = nativeDriver.video.getMaxVideoCount();
			if (videoMaxCount === 1) {
				t.skip('Only 1 video is supported');
				return;
			}
			await nativeDriver.video.clearAll();
			const video1File = await getVideoFile(VIDEO_SMALL_FILENAME);
			const video2File = await getVideoFile(VIDEO_SMALL_FILENAME);
			const video1EventEmitter = await nativeDriver.video.play(video1File.localUri, 200, 200, 430, 270);
			const video2EventEmitter = await nativeDriver.video.play(video2File.localUri, 650, 400, 430, 270);
			await Promise.all([
				new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
					video1EventEmitter.on('ended', resolve);
					video1EventEmitter.on('error', () => reject(new Error('test video play failed')));
				}),
				new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
					video2EventEmitter.on('ended', resolve);
					video2EventEmitter.on('error', () => reject(new Error('test video play failed')));
				}),
			]);
		});
		yield t.it('testPlayTooManyPostponedPlay', async () => {
			const videoMaxCount = nativeDriver.video.getMaxVideoCount();
			if (videoMaxCount === null) {
				t.skip('Max video count is not defined.');
				return;
			}

			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO_SMALL_FILENAME);
			for (let i = 0; i < videoMaxCount; i++) {
				await nativeDriver.video.play(videoFile.localUri, i * 100, 0, 100, 100);
			}
			const extraPlayPromise = nativeDriver.video.play(videoFile.localUri, videoMaxCount * 100, 0, 100, 100);
			await nativeDriver.video.stop(videoFile.localUri, 0, 0, 100, 100);
			await extraPlayPromise;
		});
		yield t.it('testPlay4KVideo', async () => {
			if (!(await nativeDriver.frontSupports(FrontCapability.VIDEO_4K))) {
				t.skip('4K video not supported');
				return;
			}
			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO_4K_FILENAME);
			await nativeDriver.video.prepare(videoFile.localUri, 0, 0, 1920, 1080, { '4k': true });
			const videoEventEmitter = await nativeDriver.video.play(videoFile.localUri, 0, 0, 1920, 1080);
			await new Promise<void>((resolve: () => void, reject: (error: Error) => void) => {
				videoEventEmitter.on('ended', resolve);
				videoEventEmitter.on('error', () => reject(new Error('test video play failed')));
			});
		});
		yield t.it('testPlay4KFailIfNotSupported', async () => {
			if (await nativeDriver.frontSupports(FrontCapability.VIDEO_4K)) {
				t.skip('4K video is supported');
				return;
			}

			await nativeDriver.video.clearAll();
			const videoFile = await getVideoFile(VIDEO_4K_FILENAME);

			await nativeDriver.video.prepare(videoFile.localUri, 0, 0, 1920, 1080, { '4k': true }).then(
				() => {
					t.skip("prepare didn't fail");
				},
				(_expected: unknown) => {
					// OK
				},
			);
		});
	});
};

async function getInternalStorageUnit() {
	const storageUnits = await nativeDriver.fileSystem.listStorageUnits();
	const storageUnit = storageUnits.find((su: IStorageUnit) => !su.removable);
	return storageUnit!;
}

async function downloadVideoFile(filename: string, uri: string) {
	const internalStorageUnit = await getInternalStorageUnit();
	const downloadPath = `${TEST_DIRECTORY_PATH}/${filename}`;
	const downloadFilePath = { storageUnit: internalStorageUnit, filePath: downloadPath };
	await nativeDriver.fileSystem.downloadFile(downloadFilePath, uri);
	const videoFile = await nativeDriver.fileSystem.getFile(downloadFilePath);
	if (!videoFile) {
		throw new Error('Failed to download video file');
	}
}

async function getVideoFile(filename: string) {
	const internalStorageUnit = await getInternalStorageUnit();
	const filePathString = `${TEST_DIRECTORY_PATH}/${filename}`;
	const filePath = { storageUnit: internalStorageUnit, filePath: filePathString };
	const videoFile = await nativeDriver.fileSystem.getFile(filePath);
	if (!videoFile) {
		throw new Error('Failed to download video file');
	}
	return videoFile;
}
