import IFrontDriver from '../../NativeDevice/Front/IFrontDriver';
import { IStorageUnit, IFilePath } from '../../NativeDevice/fileSystem';
import should from '../../Test/should';
import HashAlgorithm from '../../NativeDevice/HashAlgorithm';
import FrontCapability from '../../NativeDevice/Front/FrontCapability';
import * as t from './../../Test/TestFramework';
import { TestCase } from '@signageos/actions/dist/Device/Test/deviceTestActions';
import { NotSupportedMethodError } from '../../NativeDevice/Error/basicErrors';
import IFileSystem from '../../NativeDevice/IFileSystem';

const TEST_DIRECTORY_PATH = 'test';
const UPLOAD_URI = 'https://upload.signageos.io/upload/file?prefix=feature-tests/';
const IMAGE_URI = 'https://signageostest.blob.core.windows.net/front-display/signageos-logo_f87b2be019aa1cf704680b8984aaa5b9.png';
const TEXT_URI = 'https://signageostest.blob.core.windows.net/front-display/my-file1_c31399fd1affe0acef380c5820821af4.txt';
const TEXT2_URI = 'https://signageostest.blob.core.windows.net/front-display/my-file2_c35be2b65e97e065a4b4700106933ea7.txt';
const ARCHIVE_URI = 'https://signageostest.blob.core.windows.net/front-display/test_542d76e5d64d81483dfa9b1ef9c4a4f8.zip';
const BLUE_BACK_IMAGE_URI = 'https://2.signageos.io/assets/cars/back-black.jpg';
const CAR_ORANGE_IMAGE_URI = 'https://2.signageos.io/assets/cars/semifront-orange.jpg';
const CAR_BLACK_IMAGE_URI = 'https://2.signageos.io/assets/cars/semiside-black.jpg';
const CAR_BLUE_IMAGE_URI = 'https://2.signageos.io/assets/cars/semifront-blue.jpg';

export default async (nativeDriver: IFrontDriver) => {
	const fileSystem = nativeDriver.fileSystem;
	const internalStorageUnit: IStorageUnit | undefined = (await nativeDriver.frontSupports(FrontCapability.FILE_SYSTEM_INTERNAL_STORAGE))
		? await getInternalStorageUnit(fileSystem)
		: undefined;
	const externalStorageUnit: IStorageUnit | undefined = (await nativeDriver.frontSupports(FrontCapability.FILE_SYSTEM_EXTERNAL_STORAGE))
		? await getFirstExternalStorageUnit(fileSystem)
		: undefined;
	const supportsFileChecksum = async () => await nativeDriver.frontSupports(FrontCapability.FILE_SYSTEM_FILE_CHECKSUM);
	const supportsLink = async () => await nativeDriver.frontSupports(FrontCapability.FILE_SYSTEM_LINK);
	const supportsArchiveInfo = async () => await nativeDriver.frontSupports(FrontCapability.FILE_SYSTEM_ARCHIVE_EXTRACT_INFO);

	return t.describe(TestCase.FILE_SYSTEM, function* () {
		yield t.before('Has internal *or* external storage unit?', async () => {
			if (!internalStorageUnit && !externalStorageUnit) {
				t.skip('No.');
			}
		});

		yield t.describe('testing with internalStorageUnit', function* () {
			yield t.before('Has internal storage unit?', async () => {
				if (!internalStorageUnit) {
					t.skip('No.');
				}
			});

			const internalTestDirectory: IFilePath = {
				storageUnit: internalStorageUnit!,
				filePath: TEST_DIRECTORY_PATH,
			} as const;

			yield* manageTestDirectories(fileSystem, internalTestDirectory);

			yield t.it('should test GetFile', async () => {
				await testGetFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test GetFileNotExistsFail', async () => {
				await testGetFileNotExistsFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test WriteFile', async () => {
				await testWriteFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test WriteFileAlreadyExistsOverrides', async () => {
				await testWriteFileAlreadyExistsOverrides(fileSystem, internalTestDirectory);
			});
			yield t.it('should test AppendFile', async () => {
				await testAppendFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test AppendFileAlreadyExistsAppends', async () => {
				await testAppendFileAlreadyExistsAppends(fileSystem, internalTestDirectory);
			});
			yield t.it('should test IsDirectory', async () => {
				await testIsDirectory(fileSystem, internalTestDirectory);
			});
			yield t.it('should test IsDirectoryIfNotExistingFail', async () => {
				await testIsDirectoryIfNotExistingFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DeleteFile', async () => {
				await testDeleteFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DeleteDirectory', async () => {
				await testDeleteDirectory(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DeleteDirectoryRecursively', async () => {
				await testDeleteDirectoryRecursively(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DeleteDirectoryNotRecursivelyFail', async () => {
				await testDeleteDirectoryNotRecursivelyFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DeleteFileNotExistingFail', async () => {
				await testDeleteFileNotExistingFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test CreateDirectory', async () => {
				await testCreateDirectory(fileSystem, internalTestDirectory);
			});
			yield t.it('should test CreateDirectoryFailIfAlreadyExists', async () => {
				await testCreateDirectoryFailIfAlreadyExists(fileSystem, internalTestDirectory);
			});
			yield t.it('should test CreateDirectoryFailIfCreatingNested', async () => {
				await testCreateDirectoryFailIfCreatingNested(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DownloadFile', async () => {
				await testDownloadFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DownloadTextFile', async () => {
				await testDownloadTextFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DownloadOverridesOriginalFile', async () => {
				await testDownloadOverridesOriginalFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DownloadToNotExistingContainingDirectoryFail', async () => {
				await testDownloadToNotExistingContainingDirectoryFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test DownloadToExistingDirectoryPathFail', async () => {
				await testDownloadToExistingDirectoryPathFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test UploadFile', async () => {
				await testUploadFile(fileSystem, internalTestDirectory);
			});
			yield t.it('should test ListFiles', async () => {
				await testListFiles(fileSystem, internalTestDirectory);
			});
			yield t.it('should test ListFilesOfNotExistingDirectoryFail', async () => {
				await testListFilesOfNotExistingDirectoryFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test ListFilesOfFileFail', async () => {
				await testListFilesOfFileFail(fileSystem, internalTestDirectory);
			});
			yield t.it('should test CopyFile', async () => {
				await testCopyFile(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyDirectory', async () => {
				await testCopyDirectory(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathOverwrite', async () => {
				await testCopyFileOverwriteExistingPath(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testCopyDirectoryOverwriteExistingPath(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathFail', async () => {
				await testCopyFileToAlreadyExistingPathFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileToNotExistingContainingPathFail', async () => {
				await testCopyFileToNotExistingContainingPathFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileFromNotExistingPathFail', async () => {
				await testCopyFileFromNotExistingPathFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFile', async () => {
				await testMoveFile(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveDirectory', async () => {
				await testMoveDirectory(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathFail', async () => {
				await testMoveFileToAlreadyExistingPathFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathOverwrite', async () => {
				await testMoveFileOverwrite(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testMoveDirectoryOverwrite(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileToNotExistingContainingPathFail', async () => {
				await testMoveFileToNotExistingContainingPathFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileFromNotExistingPathFail', async () => {
				await testMoveFileFromNotExistingPathFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test LinkFile', async () => {
				await testLinkFile(fileSystem, internalTestDirectory, internalTestDirectory, supportsLink);
			});
			yield t.it('should test testLinkFileToExistingPathFail', async () => {
				await testLinkFileToExistingPathFail(fileSystem, internalTestDirectory, internalTestDirectory, supportsLink);
			});
			yield t.it('should test GetFileChecksum', async () => {
				await testGetFileChecksum(fileSystem, internalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test GetFileChecksum on problematic files', async () => {
				await testGetFileChecksumOnPictures(fileSystem, internalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test GetFileChecksumOfNotExistingFileFail', async () => {
				await testGetFileChecksumOfNotExistingFileFail(fileSystem, internalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test GetFileChecksumOfDirectoryFail', async () => {
				await testGetFileChecksumOfDirectoryFail(fileSystem, internalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test ExtractFileZip', async () => {
				await testExtractFileZip(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZipOverridesExisting', async () => {
				await testExtractFileZipOverridesExisting(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfArchiveNotExistsFail', async () => {
				await testExtractFileZipIfArchiveNotExistsFail(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfTargetNotExists', async () => {
				await testExtractFileZipIfTargetNotExists(fileSystem, internalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test GetArchiveSize', async () => {
				await testGetArchiveSize(fileSystem, internalTestDirectory, supportsArchiveInfo);
			});
			yield t.it('should test GetArchiveSizeIfArchiveNotExistsFail', async () => {
				await testGetArchiveSizeIfArchiveNotExistsFail(fileSystem, internalTestDirectory, supportsArchiveInfo);
			});
		});

		yield t.describe('testing with externalStorageUnit', function* () {
			yield t.before('Has external storage unit?', async () => {
				if (!externalStorageUnit) {
					t.skip('No.');
				}
			});

			const externalTestDirectory: IFilePath = {
				storageUnit: externalStorageUnit!,
				filePath: TEST_DIRECTORY_PATH,
			} as const;

			yield* manageTestDirectories(fileSystem, externalTestDirectory);

			yield t.it('should test GetFile', async () => {
				await testGetFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test GetFileNotExistsFail', async () => {
				await testGetFileNotExistsFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test WriteFile', async () => {
				await testWriteFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test WriteFileAlreadyExistsOverrides', async () => {
				await testWriteFileAlreadyExistsOverrides(fileSystem, externalTestDirectory);
			});
			yield t.it('should test AppendFile', async () => {
				await testAppendFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test AppendFileAlreadyExistsAppends', async () => {
				await testAppendFileAlreadyExistsAppends(fileSystem, externalTestDirectory);
			});
			yield t.it('should test IsDirectory', async () => {
				await testIsDirectory(fileSystem, externalTestDirectory);
			});
			yield t.it('should test IsDirectoryIfNotExistingFail', async () => {
				await testIsDirectoryIfNotExistingFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DeleteFile', async () => {
				await testDeleteFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DeleteDirectory', async () => {
				await testDeleteDirectory(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DeleteDirectoryRecursively', async () => {
				await testDeleteDirectoryRecursively(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DeleteDirectoryNotRecursivelyFail', async () => {
				await testDeleteDirectoryNotRecursivelyFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DeleteFileNotExistingFail', async () => {
				await testDeleteFileNotExistingFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test CreateDirectory', async () => {
				await testCreateDirectory(fileSystem, externalTestDirectory);
			});
			yield t.it('should test CreateDirectoryFailIfAlreadyExists', async () => {
				await testCreateDirectoryFailIfAlreadyExists(fileSystem, externalTestDirectory);
			});
			yield t.it('should test CreateDirectoryFailIfCreatingNested', async () => {
				await testCreateDirectoryFailIfCreatingNested(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DownloadFile', async () => {
				await testDownloadFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DownloadTextFile', async () => {
				await testDownloadTextFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DownloadOverridesOriginalFile', async () => {
				await testDownloadOverridesOriginalFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DownloadToNotExistingContainingDirectoryFail', async () => {
				await testDownloadToNotExistingContainingDirectoryFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test DownloadToExistingDirectoryPathFail', async () => {
				await testDownloadToExistingDirectoryPathFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test UploadFile', async () => {
				await testUploadFile(fileSystem, externalTestDirectory);
			});
			yield t.it('should test ListFiles', async () => {
				await testListFiles(fileSystem, externalTestDirectory);
			});
			yield t.it('should test ListFilesOfNotExistingDirectoryFail', async () => {
				await testListFilesOfNotExistingDirectoryFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test ListFilesOfFileFail', async () => {
				await testListFilesOfFileFail(fileSystem, externalTestDirectory);
			});
			yield t.it('should test GetFileChecksum', async () => {
				await testGetFileChecksum(fileSystem, externalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test GetFileChecksum on problematic files', async () => {
				await testGetFileChecksumOnPictures(fileSystem, externalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test GetFileChecksumOfNotExistingFileFail', async () => {
				await testGetFileChecksumOfNotExistingFileFail(fileSystem, externalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test GetFileChecksumOfDirectoryFail', async () => {
				await testGetFileChecksumOfDirectoryFail(fileSystem, externalTestDirectory, supportsFileChecksum);
			});
			yield t.it('should test CopyFile', async () => {
				await testCopyFile(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyDirectory', async () => {
				await testCopyDirectory(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathOverwrite', async () => {
				await testCopyFileOverwriteExistingPath(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testCopyDirectoryOverwriteExistingPath(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathFail', async () => {
				await testCopyFileToAlreadyExistingPathFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileToNotExistingContainingPathFail', async () => {
				await testCopyFileToNotExistingContainingPathFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileFromNotExistingPathFail', async () => {
				await testCopyFileFromNotExistingPathFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFile', async () => {
				await testMoveFile(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveDirectory', async () => {
				await testMoveDirectory(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathFail', async () => {
				await testMoveFileToAlreadyExistingPathFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathOverwrite', async () => {
				await testMoveFileOverwrite(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testMoveDirectoryOverwrite(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileToNotExistingContainingPathFail', async () => {
				await testMoveFileToNotExistingContainingPathFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileFromNotExistingPathFail', async () => {
				await testMoveFileFromNotExistingPathFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZip', async () => {
				await testExtractFileZip(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZipOverridesExisting', async () => {
				await testExtractFileZipOverridesExisting(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfArchiveNotExistsFail', async () => {
				await testExtractFileZipIfArchiveNotExistsFail(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfTargetNotExists', async () => {
				await testExtractFileZipIfTargetNotExists(fileSystem, externalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test GetArchiveSize', async () => {
				await testGetArchiveSize(fileSystem, externalTestDirectory, supportsArchiveInfo);
			});
			yield t.it('should test GetArchiveSizeIfArchiveNotExistsFail', async () => {
				await testGetArchiveSizeIfArchiveNotExistsFail(fileSystem, externalTestDirectory, supportsArchiveInfo);
			});
		});

		yield t.describe('testing from externalStorageUnit to internalStorageUnit', function* () {
			yield t.before('Has internal *and* external storage units?', async () => {
				if (!internalStorageUnit || !externalStorageUnit) {
					t.skip('No.');
				}
			});

			const internalTestDirectory: IFilePath = {
				storageUnit: internalStorageUnit!,
				filePath: TEST_DIRECTORY_PATH,
			} as const;
			const externalTestDirectory: IFilePath = {
				storageUnit: externalStorageUnit!,
				filePath: TEST_DIRECTORY_PATH,
			} as const;

			yield* manageTestDirectories(fileSystem, internalTestDirectory, externalTestDirectory);

			yield t.it('should test CopyFile', async () => {
				await testCopyFile(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyDirectory', async () => {
				await testCopyDirectory(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathOverwrite', async () => {
				await testCopyFileOverwriteExistingPath(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testCopyDirectoryOverwriteExistingPath(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathFail', async () => {
				await testCopyFileToAlreadyExistingPathFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileToNotExistingContainingPathFail', async () => {
				await testCopyFileToNotExistingContainingPathFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test CopyFileFromNotExistingPathFail', async () => {
				await testCopyFileFromNotExistingPathFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFile', async () => {
				await testMoveFile(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveDirectory', async () => {
				await testMoveDirectory(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathFail', async () => {
				await testMoveFileToAlreadyExistingPathFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathOverwrite', async () => {
				await testMoveFileOverwrite(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testMoveDirectoryOverwrite(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileToNotExistingContainingPathFail', async () => {
				await testMoveFileToNotExistingContainingPathFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test MoveFileFromNotExistingPathFail', async () => {
				await testMoveFileFromNotExistingPathFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZip', async () => {
				await testExtractFileZip(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZipOverridesExisting', async () => {
				await testExtractFileZipOverridesExisting(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfArchiveNotExistsFail', async () => {
				await testExtractFileZipIfArchiveNotExistsFail(fileSystem, externalTestDirectory, internalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfTargetNotExists', async () => {
				await testExtractFileZipIfTargetNotExists(fileSystem, externalTestDirectory, internalTestDirectory);
			});
		});

		yield t.describe('testing from internalStorageUnit to externalStorageUnit', function* () {
			yield t.before('Has internal *and* external storage units?', async () => {
				if (!internalStorageUnit || !externalStorageUnit) {
					t.skip('No.');
				}
			});

			const internalTestDirectory: IFilePath = {
				storageUnit: internalStorageUnit!,
				filePath: TEST_DIRECTORY_PATH,
			} as const;
			const externalTestDirectory: IFilePath = {
				storageUnit: externalStorageUnit!,
				filePath: TEST_DIRECTORY_PATH,
			} as const;

			yield* manageTestDirectories(fileSystem, internalTestDirectory, externalTestDirectory);

			yield t.it('should test CopyFile', async () => {
				await testCopyFile(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyDirectory', async () => {
				await testCopyDirectory(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathOverwrite', async () => {
				await testCopyFileOverwriteExistingPath(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testCopyDirectoryOverwriteExistingPath(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileToAlreadyExistingPathFail', async () => {
				await testCopyFileToAlreadyExistingPathFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileToNotExistingContainingPathFail', async () => {
				await testCopyFileToNotExistingContainingPathFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test CopyFileFromNotExistingPathFail', async () => {
				await testCopyFileFromNotExistingPathFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFile', async () => {
				await testMoveFile(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveDirectory', async () => {
				await testMoveDirectory(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathFail', async () => {
				await testMoveFileToAlreadyExistingPathFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileToAlreadyExistingPathOverwrite', async () => {
				await testMoveFileOverwrite(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveDirectoryToAlreadyExistingPathOverwrite', async () => {
				await testMoveDirectoryOverwrite(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileToNotExistingContainingPathFail', async () => {
				await testMoveFileToNotExistingContainingPathFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test MoveFileFromNotExistingPathFail', async () => {
				await testMoveFileFromNotExistingPathFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZip', async () => {
				await testExtractFileZip(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZipOverridesExisting', async () => {
				await testExtractFileZipOverridesExisting(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfArchiveNotExistsFail', async () => {
				await testExtractFileZipIfArchiveNotExistsFail(fileSystem, internalTestDirectory, externalTestDirectory);
			});
			yield t.it('should test ExtractFileZipIfTargetNotExists', async () => {
				await testExtractFileZipIfTargetNotExists(fileSystem, internalTestDirectory, externalTestDirectory);
			});
		});
	});
};

async function testGetFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, IMAGE_URI);
	const file = await fileSystem.getFile(downloadFilePath);
	should(file!.localUri).ok();
}

async function testGetFileNotExistsFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	const file = await fileSystem.getFile(downloadFilePath);
	should(file).null();
}

async function testWriteFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const writeFilePath = resolve(testDirectory, 'written.png');
	await fileSystem.writeFile(writeFilePath, `Some text content to write`);
	const file = await fileSystem.getFile(writeFilePath);
	should(file!.localUri).ok();

	const textResponse = await fetch(file!.localUri);
	const textContent = await textResponse.text();
	should(textContent).equal(`Some text content to write`);
}

async function testWriteFileAlreadyExistsOverrides(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const writeFilePath = resolve(testDirectory, 'written.png');

	await fileSystem.writeFile(writeFilePath, `Some text content to write`);
	const file1 = await fileSystem.getFile(writeFilePath);
	const textResponse1 = await fetch(file1!.localUri);
	const textContent1 = await textResponse1.text();
	should(textContent1).equal(`Some text content to write`);

	await fileSystem.writeFile(writeFilePath, `Rewrite with other contents`);
	const file2 = await fileSystem.getFile(writeFilePath);
	const textResponse2 = await fetch(file2!.localUri);
	const textContent2 = await textResponse2.text();
	should(textContent2).equal(`Rewrite with other contents`);
}

async function testAppendFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const appendFilePath = resolve(testDirectory, 'written.png');
	await fileSystem.appendFile(appendFilePath, `Some text content to write`);
	const file = await fileSystem.getFile(appendFilePath);
	should(file!.localUri).ok();

	const textResponse = await fetch(file!.localUri);
	const textContent = await textResponse.text();
	should(textContent).equal(`Some text content to write`);
}

async function testAppendFileAlreadyExistsAppends(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const appendFilePath = resolve(testDirectory, 'written.png');

	await fileSystem.appendFile(appendFilePath, `Some text content to write`);
	const file1 = await fileSystem.getFile(appendFilePath);
	const textResponse1 = await fetch(file1!.localUri);
	const textContent1 = await textResponse1.text();
	should(textContent1).equal(`Some text content to write`);

	await fileSystem.appendFile(appendFilePath, `\nAppend with other contents`);
	const file2 = await fileSystem.getFile(appendFilePath);
	const textResponse2 = await fetch(file2!.localUri);
	const textContent2 = await textResponse2.text();
	should(textContent2).equal(`Some text content to write\nAppend with other contents`);
}

async function testIsDirectory(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, IMAGE_URI);
	should(await fileSystem.isDirectory(testDirectory)).true();
	should(await fileSystem.isDirectory(downloadFilePath)).false();
}

async function testIsDirectoryIfNotExistingFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const testSubDirectoryFilePath = resolve(testDirectory, 'sub-dir');
	should(await fileSystem.exists(testSubDirectoryFilePath)).false();
	await should(fileSystem.isDirectory(testSubDirectoryFilePath)).rejected();
}

async function testDeleteFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, IMAGE_URI);
	should(await fileSystem.exists(downloadFilePath)).true();
	await fileSystem.deleteFile(downloadFilePath, false);
	should(await fileSystem.exists(downloadFilePath)).false();
}

async function testDeleteDirectory(fileSystem: IFileSystem, testDirectory: IFilePath) {
	should(await fileSystem.exists(testDirectory)).true();
	await fileSystem.deleteFile(testDirectory, false);
	should(await fileSystem.exists(testDirectory)).false();
}

async function testDeleteDirectoryRecursively(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, IMAGE_URI);
	should(await fileSystem.exists(testDirectory)).true();
	await fileSystem.deleteFile(testDirectory, true);
	should(await fileSystem.exists(downloadFilePath)).false();
	should(await fileSystem.exists(testDirectory)).false();
}

async function testDeleteDirectoryNotRecursivelyFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, IMAGE_URI);
	should(await fileSystem.exists(testDirectory)).true();
	await should(fileSystem.deleteFile(testDirectory, false)).rejected();
	should(await fileSystem.exists(downloadFilePath)).true();
	should(await fileSystem.exists(testDirectory)).true();
}

async function testDeleteFileNotExistingFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	should(await fileSystem.exists(downloadFilePath)).false();
	await should(fileSystem.deleteFile(downloadFilePath, false)).rejected();
	await should(fileSystem.deleteFile(downloadFilePath, true)).rejected();
}

async function testCreateDirectory(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const newDirectory = resolve(testDirectory, 'new-directory');
	await fileSystem.createDirectory(newDirectory);
	should(await fileSystem.exists(newDirectory)).true();
	should(await fileSystem.isDirectory(newDirectory)).true();
}

async function testCreateDirectoryFailIfAlreadyExists(fileSystem: IFileSystem, testDirectory: IFilePath) {
	await should(fileSystem.createDirectory(testDirectory)).rejected();
}

async function testCreateDirectoryFailIfCreatingNested(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const newDirectory = resolve(testDirectory, 'absent/nested');

	await should(fileSystem.createDirectory(newDirectory)).rejected();
}

async function testDownloadFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, IMAGE_URI);
	should(await fileSystem.exists(downloadFilePath)).true();
	should(await fileSystem.isDirectory(downloadFilePath)).false();
}

async function testDownloadTextFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, TEXT_URI);
	should(await fileSystem.exists(downloadFilePath)).true();
	should(await fileSystem.isDirectory(downloadFilePath)).false();

	const textFile = await fileSystem.getFile(downloadFilePath);

	const textResponse = await fetch(textFile!.localUri);
	const textContent = await textResponse.text();
	should(textContent).equal('content-1\n');
}

async function testDownloadOverridesOriginalFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, TEXT_URI);

	const firstFile = await fileSystem.getFile(downloadFilePath);
	const firstResponse = await fetch(firstFile!.localUri);
	const firstContent = await firstResponse.text();
	should(firstContent).equal('content-1\n');

	await fileSystem.downloadFile(downloadFilePath, TEXT2_URI);
	const secondFile = await fileSystem.getFile(downloadFilePath);
	const secondResponse = await fetch(secondFile!.localUri);
	const secondContent = await secondResponse.text();
	should(secondContent).equal('content-2\n');
}

async function testDownloadToNotExistingContainingDirectoryFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'sub-directory/download.png');
	await should(fileSystem.downloadFile(downloadFilePath, TEXT_URI)).rejected();
}

async function testDownloadToExistingDirectoryPathFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const folder = resolve(testDirectory, 'folder');
	await fileSystem.createDirectory(folder);
	await should(fileSystem.downloadFile(folder, TEXT_URI)).rejected();
}

async function testUploadFile(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const uploadFilePath = resolve(testDirectory, 'upload.txt');
	await fileSystem.writeFile(uploadFilePath, 'test');
	const response = await fileSystem.uploadFile(uploadFilePath, UPLOAD_URI, 'file');
	const remoteFileUri: string = response && JSON.parse(response).uri;
	should(remoteFileUri).String();
	const remoteResponse = await fetch(remoteFileUri);
	const remoteContent = await remoteResponse.text();
	should(remoteContent).equal('test');
}

async function testListFiles(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const file1 = resolve(testDirectory, 'file1');
	const file2 = resolve(testDirectory, 'file2');
	const folder = resolve(testDirectory, 'folder');
	const file3 = resolve(folder, 'file3');

	await fileSystem.createDirectory(folder);
	await Promise.all([
		fileSystem.downloadFile(file1, IMAGE_URI),
		fileSystem.downloadFile(file2, TEXT_URI),
		fileSystem.downloadFile(file3, ARCHIVE_URI),
	]);
	const testFilePaths = await fileSystem.listFiles(testDirectory);
	const filePathSortFunction = (a: IFilePath, b: IFilePath) => a.filePath.localeCompare(b.filePath);
	should(testFilePaths.sort(filePathSortFunction)).eql([file1, file2, folder]);
	const testFolderFilePaths = await fileSystem.listFiles(folder);
	should(testFolderFilePaths).eql([file3]);
}

async function testListFilesOfNotExistingDirectoryFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const absentDirectory = resolve(testDirectory, 'absent');

	await should(fileSystem.listFiles(absentDirectory)).rejected();
}

async function testListFilesOfFileFail(fileSystem: IFileSystem, testDirectory: IFilePath) {
	const downloadFilePath = resolve(testDirectory, 'download.png');
	await fileSystem.downloadFile(downloadFilePath, TEXT_URI);

	await should(fileSystem.listFiles(downloadFilePath)).rejected();
}

async function testCopyFile(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	await fileSystem.copyFile(fromFilePath, toFilePath);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
}

async function testCopyDirectory(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromSubDirectoryPath = resolve(fromTestDirectory, 'dir');
	const fromFilePath = resolve(fromSubDirectoryPath, 'file1');
	const toFilePath = resolve(toTestDirectory, 'dir2');
	const toSubFilePath = resolve(toFilePath, 'file1');
	await fileSystem.createDirectory(fromSubDirectoryPath);
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	should(await fileSystem.exists(toSubFilePath)).false();
	await fileSystem.copyFile(fromSubDirectoryPath, toFilePath);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.isDirectory(toFilePath)).true();
	should(await fileSystem.exists(toSubFilePath)).true();
	should(await fileSystem.isDirectory(toSubFilePath)).false();
}

async function testCopyFileOverwriteExistingPath(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.writeFile(fromFilePath, 'source');
	await fileSystem.writeFile(toFilePath, 'destination');
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	await fileSystem.copyFile(fromFilePath, toFilePath, { overwrite: true });
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.readFile(toFilePath)).equal('source');
}

async function testCopyDirectoryOverwriteExistingPath(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromSubDirectoryPath = resolve(fromTestDirectory, 'dir');
	const fromFilePath = resolve(fromSubDirectoryPath, 'file1');
	const toFilePath = resolve(toTestDirectory, 'dir2');
	const toSubFilePath = resolve(toFilePath, 'file1');
	await fileSystem.createDirectory(fromSubDirectoryPath);
	await fileSystem.writeFile(fromFilePath, 'source');
	await fileSystem.createDirectory(toFilePath);
	await fileSystem.writeFile(toSubFilePath, 'destination');
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.exists(toSubFilePath)).true();
	await fileSystem.copyFile(fromSubDirectoryPath, toFilePath, { overwrite: true });
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.isDirectory(toFilePath)).true();
	should(await fileSystem.exists(toSubFilePath)).true();
	should(await fileSystem.isDirectory(toSubFilePath)).false();
	should(await fileSystem.readFile(toSubFilePath)).equal('source');
}

async function testCopyFileToAlreadyExistingPathFail(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	await fileSystem.downloadFile(toFilePath, TEXT_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	await should(fileSystem.copyFile(fromFilePath, toFilePath)).rejected();
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
}

async function testCopyFileToNotExistingContainingPathFail(
	fileSystem: IFileSystem,
	fromTestDirectory: IFilePath,
	toTestDirectory: IFilePath,
) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'absent/file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	await should(fileSystem.copyFile(fromFilePath, toFilePath)).rejected();
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
}

async function testCopyFileFromNotExistingPathFail(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).false();
	await should(fileSystem.copyFile(fromFilePath, toFilePath)).rejected();
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).false();
}

async function testMoveFile(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	await fileSystem.moveFile(fromFilePath, toFilePath);
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).true();
}

async function testMoveDirectory(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromSubDirectoryPath = resolve(fromTestDirectory, 'dir');
	const fromFilePath = resolve(fromSubDirectoryPath, 'file1');
	const toFilePath = resolve(toTestDirectory, 'dir2');
	const toSubFilePath = resolve(toFilePath, 'file1');
	await fileSystem.createDirectory(fromSubDirectoryPath);
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	should(await fileSystem.exists(toSubFilePath)).false();
	await fileSystem.moveFile(fromSubDirectoryPath, toFilePath);
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.isDirectory(toFilePath)).true();
	should(await fileSystem.exists(toSubFilePath)).true();
	should(await fileSystem.isDirectory(toSubFilePath)).false();
}

async function testMoveFileToAlreadyExistingPathFail(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	await fileSystem.downloadFile(toFilePath, TEXT_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	await should(fileSystem.moveFile(fromFilePath, toFilePath)).rejected();
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
}

async function testMoveFileOverwrite(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.writeFile(fromFilePath, 'source');
	await fileSystem.writeFile(toFilePath, 'destination');
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	await fileSystem.moveFile(fromFilePath, toFilePath, { overwrite: true });
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.readFile(toFilePath)).equal('source');
}

async function testMoveDirectoryOverwrite(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromSubDirectoryPath = resolve(fromTestDirectory, 'dir');
	const fromFilePath = resolve(fromSubDirectoryPath, 'file1');
	const toFilePath = resolve(toTestDirectory, 'dir2');
	const toSubFilePath = resolve(toFilePath, 'file1');
	await fileSystem.createDirectory(fromSubDirectoryPath);
	await fileSystem.createDirectory(toFilePath);
	await fileSystem.writeFile(fromFilePath, 'source');
	await fileSystem.writeFile(toSubFilePath, 'destination');
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.exists(toSubFilePath)).true();
	await fileSystem.moveFile(fromSubDirectoryPath, toFilePath, { overwrite: true });
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).true();
	should(await fileSystem.isDirectory(toFilePath)).true();
	should(await fileSystem.exists(toSubFilePath)).true();
	should(await fileSystem.isDirectory(toSubFilePath)).false();
	should(await fileSystem.readFile(toSubFilePath)).equal('source');
}

async function testMoveFileToNotExistingContainingPathFail(
	fileSystem: IFileSystem,
	fromTestDirectory: IFilePath,
	toTestDirectory: IFilePath,
) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'dir1/file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	await should(fileSystem.moveFile(fromFilePath, toFilePath)).rejected();
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
}

async function testMoveFileFromNotExistingPathFail(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).false();
	await should(fileSystem.moveFile(fromFilePath, toFilePath)).rejected();
	should(await fileSystem.exists(fromFilePath)).false();
	should(await fileSystem.exists(toFilePath)).false();
}

async function testLinkFile(
	fileSystem: IFileSystem,
	fromTestDirectory: IFilePath,
	toTestDirectory: IFilePath,
	supportsLink: () => Promise<boolean>,
) {
	if (!(await supportsLink())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_LINK]} capability`);
		return;
	}

	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).false();
	await fileSystem.link(fromFilePath, toFilePath);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
}

async function testLinkFileToExistingPathFail(
	fileSystem: IFileSystem,
	fromTestDirectory: IFilePath,
	toTestDirectory: IFilePath,
	supportsLink: () => Promise<boolean>,
) {
	if (!(await supportsLink())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_LINK]} capability`);
		return;
	}

	const fromFilePath = resolve(fromTestDirectory, 'file1');
	const toFilePath = resolve(toTestDirectory, 'file2');
	await fileSystem.downloadFile(fromFilePath, IMAGE_URI);
	await fileSystem.downloadFile(toFilePath, TEXT_URI);
	should(await fileSystem.exists(fromFilePath)).true();
	should(await fileSystem.exists(toFilePath)).true();
	let rejected = false;
	try {
		await fileSystem.link(fromFilePath, toFilePath);
	} catch (error) {
		if (error instanceof NotSupportedMethodError) {
			throw error;
		}
		rejected = true;
	}

	if (!rejected) {
		throw new Error('link was expected to be rejected but it resolved');
	}
}

async function testGetFileChecksum(fileSystem: IFileSystem, testDirectory: IFilePath, supportsFileChecksum: () => Promise<boolean>) {
	if (!(await supportsFileChecksum())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_FILE_CHECKSUM]} capability`);
		return;
	}

	const checksumFilePath = resolve(testDirectory, 'checksum-file');
	await fileSystem.downloadFile(checksumFilePath, IMAGE_URI);
	should(await fileSystem.getFileChecksum(checksumFilePath, HashAlgorithm.MD5)).equal('f87b2be019aa1cf704680b8984aaa5b9');
}

async function testGetFileChecksumOnPictures(
	fileSystem: IFileSystem,
	testDirectory: IFilePath,
	supportsFileChecksum: () => Promise<boolean>,
) {
	if (!(await supportsFileChecksum())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_FILE_CHECKSUM]} capability`);
		return;
	}

	const checksumFilePathBlack = resolve(testDirectory, 'checksum-file-' + 'CAR_BLACK_IMAGE_URI');
	await fileSystem.downloadFile(checksumFilePathBlack, CAR_BLACK_IMAGE_URI, { 'checksum-crc32': 'enabled' });
	should(await fileSystem.getFileChecksum(checksumFilePathBlack, HashAlgorithm.CRC32)).equal('056fa2ca');

	const checksumFilePathBlue = resolve(testDirectory, 'checksum-file-' + 'CAR_BLUE_IMAGE_URI');
	await fileSystem.downloadFile(checksumFilePathBlue, CAR_BLUE_IMAGE_URI, { 'checksum-crc32': 'enabled' });
	should(await fileSystem.getFileChecksum(checksumFilePathBlue, HashAlgorithm.CRC32)).equal('9a648a53');

	const checksumFilePathOrange = resolve(testDirectory, 'checksum-file-' + 'CAR_ORANGE_IMAGE_URI');
	await fileSystem.downloadFile(checksumFilePathOrange, CAR_ORANGE_IMAGE_URI, { 'checksum-crc32': 'enabled' });
	should(await fileSystem.getFileChecksum(checksumFilePathOrange, HashAlgorithm.CRC32)).equal('7d7911ee');

	const checksumFilePathBlueBack = resolve(testDirectory, 'checksum-file-' + 'BLUE_BACK_IMAGE_URI');
	await fileSystem.downloadFile(checksumFilePathBlueBack, BLUE_BACK_IMAGE_URI, { 'checksum-crc32': 'enabled' });
	should(await fileSystem.getFileChecksum(checksumFilePathBlueBack, HashAlgorithm.CRC32)).equal('4d60de84');
}

async function testGetFileChecksumOfNotExistingFileFail(
	fileSystem: IFileSystem,
	testDirectory: IFilePath,
	supportsFileChecksum: () => Promise<boolean>,
) {
	if (!(await supportsFileChecksum())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_FILE_CHECKSUM]} capability`);
		return;
	}

	const checksumFilePath = resolve(testDirectory, 'unknown-checksum-file');
	await should(fileSystem.getFileChecksum(checksumFilePath, HashAlgorithm.MD5)).rejected();
}

async function testGetFileChecksumOfDirectoryFail(
	fileSystem: IFileSystem,
	testDirectory: IFilePath,
	supportsFileChecksum: () => Promise<boolean>,
) {
	if (!(await supportsFileChecksum())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_FILE_CHECKSUM]} capability`);
		return;
	}

	await should(fileSystem.getFileChecksum(testDirectory, HashAlgorithm.MD5)).rejected();
}

async function testExtractFileZip(
	fileSystem: IFileSystem,
	fromTestDirectory: IFilePath,
	toTestDirectory: IFilePath,
	createOutputDirectory: boolean = true,
) {
	const archiveFilePath = resolve(fromTestDirectory, 'archive.zip');
	const extractDirectoryFilePath = resolve(toTestDirectory, 'extracted');

	if (createOutputDirectory) {
		await fileSystem.createDirectory(extractDirectoryFilePath);
	}

	await fileSystem.downloadFile(archiveFilePath, ARCHIVE_URI);
	await fileSystem.extractFile(archiveFilePath, extractDirectoryFilePath, 'zip');
	const file1FilePath = resolve(extractDirectoryFilePath, 'my-file1');
	const file2FilePath = resolve(extractDirectoryFilePath, 'my-file2');
	const folderFilePath = resolve(extractDirectoryFilePath, 'my-folder');
	const file3FilePath = resolve(folderFilePath, 'my-file3');
	const subFolderFilePath = resolve(folderFilePath, 'sub-folder');
	const file4FilePath = resolve(subFolderFilePath, 'my-file4');
	should(await fileSystem.exists(file1FilePath)).true();
	should(await fileSystem.isDirectory(file1FilePath)).false();
	should(await fileSystem.exists(file2FilePath)).true();
	should(await fileSystem.isDirectory(file2FilePath)).false();
	should(await fileSystem.exists(folderFilePath)).true();
	should(await fileSystem.isDirectory(folderFilePath)).true();
	should(await fileSystem.exists(file3FilePath)).true();
	should(await fileSystem.isDirectory(file3FilePath)).false();
	should(await fileSystem.exists(subFolderFilePath)).true();
	should(await fileSystem.isDirectory(subFolderFilePath)).true();
	should(await fileSystem.exists(file4FilePath)).true();
	should(await fileSystem.isDirectory(file4FilePath)).false();

	const file1 = await fileSystem.getFile(file1FilePath);
	const text1Response = await fetch(file1!.localUri);
	const text1Content = await text1Response.text();
	should(text1Content).equal('content-1\n');

	const file2 = await fileSystem.getFile(file2FilePath);
	const text2Response = await fetch(file2!.localUri);
	const text2Content = await text2Response.text();
	should(text2Content).equal('content-2\n');

	const file3 = await fileSystem.getFile(file3FilePath);
	const text3Response = await fetch(file3!.localUri);
	const text3Content = await text3Response.text();
	should(text3Content).equal('content-3\n');

	const file4 = await fileSystem.getFile(file4FilePath);
	const text4Response = await fetch(file4!.localUri);
	const text4Content = await text4Response.text();
	should(text4Content).equal('content-4\n');
}

async function testExtractFileZipOverridesExisting(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const extractDirectoryFilePath = resolve(toTestDirectory, 'extracted');
	const file1FilePath = resolve(extractDirectoryFilePath, 'my-file1');
	const file2FilePath = resolve(extractDirectoryFilePath, 'my-file2');
	const folderFilePath = resolve(extractDirectoryFilePath, 'my-folder');
	const file3FilePath = resolve(folderFilePath, 'my-file3');
	const subFolderFilePath = resolve(folderFilePath, 'sub-folder');
	const file4FilePath = resolve(subFolderFilePath, 'my-file4');
	await fileSystem.createDirectory(extractDirectoryFilePath);
	await fileSystem.writeFile(file1FilePath, 'bad');
	await fileSystem.writeFile(file2FilePath, 'bad');
	await fileSystem.createDirectory(folderFilePath);
	await fileSystem.writeFile(file3FilePath, 'bad');
	await fileSystem.createDirectory(subFolderFilePath);
	await fileSystem.writeFile(file4FilePath, 'bad');

	await testExtractFileZip(fileSystem, fromTestDirectory, toTestDirectory, false);
}

async function testExtractFileZipIfArchiveNotExistsFail(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const archiveFilePath = resolve(fromTestDirectory, 'archive.zip');
	const extractDirectoryFilePath = resolve(toTestDirectory, 'extracted');
	await fileSystem.createDirectory(extractDirectoryFilePath);
	await should(fileSystem.extractFile(archiveFilePath, extractDirectoryFilePath, 'zip')).rejected();
}

async function testExtractFileZipIfTargetNotExists(fileSystem: IFileSystem, fromTestDirectory: IFilePath, toTestDirectory: IFilePath) {
	const archiveFilePath = resolve(fromTestDirectory, 'archive.zip');
	const extractDirectoryFilePath = resolve(toTestDirectory, 'extracted');
	await fileSystem.downloadFile(archiveFilePath, ARCHIVE_URI);
	await fileSystem.extractFile(archiveFilePath, extractDirectoryFilePath, 'zip');
}

async function testGetArchiveSize(fileSystem: IFileSystem, testDirectory: IFilePath, supportsArchiveInfo: () => Promise<boolean>) {
	if (!(await supportsArchiveInfo())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_ARCHIVE_EXTRACT_INFO]} capability`);
		return;
	}
	const archiveFilePath = resolve(testDirectory, 'archive.zip');
	await fileSystem.downloadFile(archiveFilePath, ARCHIVE_URI);
	const { uncompressedSize } = await fileSystem.getArchiveInfo(archiveFilePath);
	should(uncompressedSize).be.eql(40);
}

async function testGetArchiveSizeIfArchiveNotExistsFail(
	fileSystem: IFileSystem,
	testDirectory: IFilePath,
	supportsArchiveInfo: () => Promise<boolean>,
) {
	if (!(await supportsArchiveInfo())) {
		t.skip(`Device doesn't provide ${FrontCapability[FrontCapability.FILE_SYSTEM_ARCHIVE_EXTRACT_INFO]} capability`);
		return;
	}
	const archiveFilePath = resolve(testDirectory, 'unknown_archive.zip');
	await should(fileSystem.getArchiveInfo(archiveFilePath)).rejected();
}

async function getInternalStorageUnit(fileSystem: IFileSystem) {
	const storageUnits = await fileSystem.listStorageUnits();
	const storageUnit = storageUnits.find((su: IStorageUnit) => !su.removable);
	return storageUnit!;
}

async function getFirstExternalStorageUnit(fileSystem: IFileSystem) {
	const storageUnits = await fileSystem.listStorageUnits();
	const storageUnit = storageUnits.find((su: IStorageUnit) => su.removable);
	return storageUnit;
}

async function createDirectories(fileSystem: IFileSystem, ...filePaths: IFilePath[]) {
	await Promise.all(
		filePaths.map(async (filePath: IFilePath) => {
			await fileSystem.createDirectory(filePath);
		}),
	);
}

async function deleteRecursivelyIfExists(fileSystem: IFileSystem, ...filePaths: IFilePath[]) {
	await Promise.all(
		filePaths.map(async (filePath: IFilePath) => {
			if (await fileSystem.exists(filePath)) {
				await fileSystem.deleteFile(filePath, true);
			}
		}),
	);
}

function* manageTestDirectories(fileSystem: IFileSystem, ...filePaths: IFilePath[]) {
	yield t.beforeEach('clear test directory', async () => {
		await deleteRecursivelyIfExists(fileSystem, ...filePaths);
		await createDirectories(fileSystem, ...filePaths);
	});
	yield t.afterEach('clear test directory', async () => {
		await deleteRecursivelyIfExists(fileSystem, ...filePaths);
	});
}

function resolve(base: IFilePath, filePath: string): IFilePath {
	if (filePath.startsWith('/')) {
		return {
			...base,
			filePath,
		};
	} else if (filePath.startsWith('.')) {
		throw new Error('Not implemented');
	} else {
		return {
			...base,
			filePath: base.filePath + '/' + filePath,
		};
	}
}
