import { EventEmitter } from 'events';
import IPinStorage from './IPinStorage';
import ISecurity from './ISecurity';

export default class DefaultSecurity implements ISecurity {
	private readonly events: EventEmitter = new EventEmitter();

	constructor(
		private readonly storage: IPinStorage,
		private readonly setPinDelegate: (pin: string) => Promise<void>,
	) {}

	public async isSecure(): Promise<boolean> {
		return (await this.getPin()) !== '';
	}

	public async verifyPin(pin: string): Promise<boolean> {
		const currentPin = await this.getPin();
		if (currentPin !== '' && currentPin === pin) {
			return true;
		}
		// PIN doesn't match or no PIN is set.
		return false;
	}

	public async getPin(): Promise<string> {
		return (await this.storage.loadPin()) ?? '';
	}

	public async setPin(newPin: string): Promise<void> {
		const oldPin = await this.storage.loadPin();
		await this.storage.savePin(newPin);
		try {
			await this.setPinDelegate(newPin);
			this.events.emit('pin_set');
		} catch (e) {
			// Restore old PIN on failure.
			await this.storage.savePin(oldPin);
			throw e;
		}
	}

	public onPinSet(listener: () => void): () => void {
		this.events.addListener('pin_set', listener);
		return () => this.events.removeListener('pin_set', listener);
	}
}
