import { StartApplication } from '@signageos/actions/dist/Application/applicationActions';
import { UpdateDeviceTelemetryRecord } from '@signageos/actions/dist/Device/Telemetry/deviceTelemetryActions';
import { UpdateDeviceName } from '@signageos/actions/dist/Device/deviceActions';
import { GetLocation, ILocation, UpdateLocation } from '@signageos/actions/dist/Location/locationActions';
import {
	GetOrganizationTags,
	IOrganizationTag,
	UpdateOrganizationTags,
} from '@signageos/actions/dist/Organization/Tag/organizationTagActions';
import { UpdateOrganization } from '@signageos/actions/dist/Organization/organizationActions';
import { MonitoringLogData } from '@signageos/common-types/dist/Device/MonitoringLog/MonitoringLogData';
import { DeviceTelemetryType } from '@signageos/common-types/dist/Device/Telemetry/DeviceTelemetryType';
import { isEqual } from 'lodash';
import { PutEffect, put, takeEvery } from 'redux-saga/effects';
import { notifyDeviceApplicationVersionCommon, notifyDeviceInfo, notifyDeviceProxyInfo } from '../../Device/deviceSagas';
import FrontCapability from '../../NativeDevice/Front/FrontCapability';
import IFrontDriver from '../../NativeDevice/Front/IFrontDriver';
import IBasicDriver from '../../NativeDevice/IBasicDriver';
import { IFrontManagementDriver } from '../../NativeDevice/Management/IManagementDriver';
import Property from '../../Property/Property';
import { IPropertyStorage } from '../../Property/propertyStorage';
import { bindAndTakeEvery, bindWhenPlatform } from '../../Socket/socketActionCreator';
import { deliver } from '../../Socket/socketActionDeliverHelper';
import { UpdateDisplayFirmwareVersion } from './deviceActions';

export function* updateTitle(window: Window) {
	yield bindAndTakeEvery(UpdateDeviceName, function* (action: UpdateDeviceName): IterableIterator<any> {
		try {
			const titleElement = window.document.head.getElementsByTagName('title')[0];
			titleElement.innerHTML = action.deviceName;
		} catch (error) {
			console.error('updateTitle failed', error);
		}
	});
	// For displaying organization in default screen
	yield bindWhenPlatform(UpdateOrganization);
}

export function* notifyDeviceSaga(getNativeDriver: () => IFrontManagementDriver) {
	yield takeEvery(StartApplication, notifyDeviceInfo, getNativeDriver);
	yield takeEvery(StartApplication, notifyDeviceProxyInfo, getNativeDriver);
	yield takeEvery(StartApplication, notifyFirmwareVersion(getNativeDriver));
}

export function* notifyDeviceApplicationVersion(getNativeDriver: () => IBasicDriver, applicationVersion: string) {
	yield takeEvery(StartApplication, notifyDeviceApplicationVersionCommon, getNativeDriver, applicationVersion);
}

export function* updateDeviceLocation(propertyStorage: IPropertyStorage) {
	yield bindAndTakeEvery(UpdateLocation, function* (action: UpdateLocation) {
		yield propertyStorage.setValue<ILocation | null>(Property.DEVICE_LOCATION, action.location);
	});

	yield takeEvery(StartApplication, function* (): Generator<Promise<ILocation | null> | PutEffect, void, ILocation | null> {
		const storedLocation = yield propertyStorage.getValueOrDefault<ILocation | null>(Property.DEVICE_LOCATION, null);
		yield put({
			type: UpdateLocation,
			location: storedLocation,
		});
	});
}

export function* updateOrganizationTags(propertyStorage: IPropertyStorage) {
	yield bindAndTakeEvery(UpdateOrganizationTags, function* (action: UpdateOrganizationTags) {
		yield propertyStorage.setValue<IOrganizationTag[]>(Property.DEVICE_ORGANIZATION_TAGS, action.tags);
	});

	yield takeEvery(StartApplication, function* (): Generator<Promise<IOrganizationTag[]> | PutEffect, void, IOrganizationTag[]> {
		const storedOrganizationTags = yield propertyStorage.getValueOrDefault<IOrganizationTag[]>(Property.DEVICE_ORGANIZATION_TAGS, []);
		yield put({
			type: UpdateOrganizationTags,
			tags: storedOrganizationTags,
		});
	});
}

/**
 * Update device location and organization tags in local storage
 * which can be later retrieved by the JS API.
 */
export function* getDeviceLocationAndTags() {
	yield takeEvery(StartApplication, function* () {
		yield put({ type: GetLocation });
		yield put({ type: GetOrganizationTags });
	});
}

export function* notifyFrontCapabilities(propertyStorage: IPropertyStorage, getNativeDriver: () => IFrontDriver) {
	let supportedCapabilities: FrontCapability[] = [];
	try {
		const nativeDriver = getNativeDriver();
		for (const capability of Object.values(FrontCapability)) {
			const supported: boolean = yield nativeDriver.frontSupports(capability);
			if (supported) {
				supportedCapabilities.push(capability);
			}
		}
		const isCapable: MonitoringLogData[DeviceTelemetryType.FRONT_CAPABILITIES] = {
			capable: supportedCapabilities,
		};

		const lastUpdated: MonitoringLogData[DeviceTelemetryType.FRONT_CAPABILITIES] = yield propertyStorage.getValueOrDefault(
			Property.LATEST_REPORTED_SETTINGS_FRONT_CAPABILITIES,
			[],
		);

		if (!isEqual(lastUpdated, isCapable)) {
			yield propertyStorage.setValue(Property.LATEST_REPORTED_SETTINGS_FRONT_CAPABILITIES, isCapable);
			yield deliver<
				UpdateDeviceTelemetryRecord<DeviceTelemetryType.FRONT_CAPABILITIES, MonitoringLogData[DeviceTelemetryType.FRONT_CAPABILITIES]>
			>({
				type: UpdateDeviceTelemetryRecord,
				name: DeviceTelemetryType.FRONT_CAPABILITIES,
				data: isCapable,
			});
		}
	} catch (error) {
		console.error('notifyCapabilities failed front', error);
	}
}

export const notifyFirmwareVersion = (getManagementDriver: () => IFrontManagementDriver) => {
	return function* () {
		try {
			const firmwareVersion: string = yield getManagementDriver().firmwareGetVersion();
			yield put({ type: UpdateDisplayFirmwareVersion, version: firmwareVersion });
		} catch (error) {
			console.error('notifyFirmwareVersion failed', error);
		}
	};
};
