import { bindAndTakeEvery, takeEveryAndBindWhenPlatform } from '../../Socket/socketActionCreator';
import { put, takeEvery } from 'redux-saga/effects';
import { now } from '@signageos/lib/dist/DateTime/dateTimeFactory';
import {
	IAppletTimingDefinition,
	LastActiveAppletMissed,
	UpdateActiveAppletBinary,
	UpdateActiveAppletUniqueHash,
	UpdateAppletTimingsDefinition,
} from '@signageos/actions/dist/Applet/appletActions';
import { HandleMotion } from '@signageos/actions/dist/Input/motionActions';
import { HandleKeyUp } from '@signageos/actions/dist/Input/keyActions';
import { IWebWorkerFactory } from '../../WebWorker/masterWebWorkerFactory';
import {
	HideApplet,
	HideOSD,
	PauseAppletTimingTriggers,
	PrepareNextAppletTiming,
	ResumeAppletTimingTriggers,
	ShowNextAppletTiming,
	ShowOSD,
} from './appletTimingActions';
import { createChannel, takeEvery as takeEveryChannelMessage } from '../../ReduxSaga/channels';
import { DisconnectDevice } from '@signageos/actions/dist/Device/Connect/deviceConnectActions';
import { ActiveAppletRestore } from '../Applet/activeAppletActions';
import { PerformPowerAction } from '@signageos/actions/dist/Device/Power/devicePowerActions';
import PowerActionType from '@signageos/actions/dist/Device/Power/PowerActionType';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import { checksumString } from '@signageos/lib/dist/Hash/checksum';
import TimingFinishEventType from './TimingFinishEventType';
import { DeviceConnected, IDeviceConnected } from '../../Device/Connect/deviceConnectActions';

function createConnectedAppletTimingDefinition(appletUid: string, appletVersion: string): IAppletTimingDefinition {
	return {
		appletUid,
		configuration: {},
		timmingChecksum: checksumString(appletUid + ',' + appletVersion),
		appletVersion,
		isPackage: true,
		appletVersionPostfix: generateUniqueHash(),
		finishEvent: {
			type: TimingFinishEventType[TimingFinishEventType.DURATION],
			data: null,
		},
		position: 1,
		metadata: {},
	};
}

/**
 * This saga controls the flow of when to run which applet
 *
 * It starts with no definitions and in that case, there are no applets to run so it does nothing.
 *
 * Once it receives action UpdateAppletTimingsDefinition with new timing definitions, it stores the definitions
 * and starts processing them from the beginning.
 *
 * The heart of this process is an infinite loop that iterates over the definitions list and triggers switch to the next applet
 * when conditions are met. However, this loop is not part of this file. Since it's an infinite loop whose only job is to do some
 * simple decision making, it made sense to run it in a separate service worker - i.e. another thread.
 * This infinite loop can be found in src/Front/AppletTiming/AppletTimingController.ts.
 */
export function* controlActiveAppletTiming(webWorkerFactory: IWebWorkerFactory, defaultDefinitions?: IAppletTimingDefinition[]) {
	const defaultDefinitionsOrNull = (): IAppletTimingDefinition[] | null => {
		if (defaultDefinitions && defaultDefinitions.length > 0) {
			return defaultDefinitions;
		}
		return null;
	};

	const appletTimingWebWorker = webWorkerFactory.createAppletTimingController();
	const appletTimingChannel = createChannel<ShowNextAppletTiming>((putHere: (message: ShowNextAppletTiming) => void) => {
		appletTimingWebWorker.onMessage(putHere);
	});

	appletTimingWebWorker.start(undefined);

	let model: ILoopModel;

	const setModel = (newModel: ILoopModel) => {
		model = newModel;
		appletTimingWebWorker.postMessage(newModel);
	};

	setModel({
		definitions: defaultDefinitionsOrNull(),
		currentDefinitionIndex: null,
		startTimestamp: null,
		lastInteractionTimestamp: null,
		triggersPausedTimestamp: null,
		lastPutUpdateActiveAppletBinary: null,
		lastPutPrepareNextAppletTiming: null,
		deviceInConnectedMode: null,
	});

	yield takeEveryAndBindWhenPlatform(
		UpdateAppletTimingsDefinition,
		function* (action: UpdateAppletTimingsDefinition): IterableIterator<any> {
			let definitions: IAppletTimingDefinition[] | null;
			if (action.definitions && action.definitions.length > 0) {
				definitions = action.definitions.sort((a: IAppletTimingDefinition, b: IAppletTimingDefinition) => a.position - b.position);
			} else {
				definitions = defaultDefinitionsOrNull();
			}
			setModel({
				...model,
				definitions,
				currentDefinitionIndex: null,
				startTimestamp: null,
				lastInteractionTimestamp: null,
				triggersPausedTimestamp: null,
				lastPutUpdateActiveAppletBinary: null,
				lastPutPrepareNextAppletTiming: null,
			});
		},
	);
	yield bindAndTakeEvery(LastActiveAppletMissed, function* (): IterableIterator<any> {
		const definitions = defaultDefinitionsOrNull();
		setModel({
			...model,
			definitions,
			currentDefinitionIndex: null,
			startTimestamp: null,
			lastInteractionTimestamp: null,
			triggersPausedTimestamp: null,
			lastPutUpdateActiveAppletBinary: null,
			lastPutPrepareNextAppletTiming: null,
		});
		if (!defaultDefinitions) {
			yield put<HideApplet>({
				type: HideApplet,
			});
		}
	});
	yield takeEvery([HandleMotion, HandleKeyUp], function* (): IterableIterator<any> {
		if (model.currentDefinitionIndex !== null && model.triggersPausedTimestamp === null) {
			setModel({
				...model,
				lastInteractionTimestamp: now().valueOf() as number,
			});
		}
	});
	yield takeEvery([PauseAppletTimingTriggers, ShowOSD], function* (): IterableIterator<any> {
		if (model.currentDefinitionIndex !== null && model.triggersPausedTimestamp === null) {
			setModel({
				...model,
				triggersPausedTimestamp: now().valueOf() as number,
			});
		}
	});
	yield takeEvery([ResumeAppletTimingTriggers, UpdateActiveAppletUniqueHash], function* (): IterableIterator<any> {
		if (model.startTimestamp !== null && model.triggersPausedTimestamp !== null) {
			const pauseDuration = now().valueOf() - model.triggersPausedTimestamp;
			setModel({
				...model,
				startTimestamp: model.startTimestamp + pauseDuration,
				triggersPausedTimestamp: null,
			});
		}
	});
	yield takeEvery(HideOSD, function* (): IterableIterator<any> {
		setModel({
			...model,
			currentDefinitionIndex: null,
			startTimestamp: null,
			lastInteractionTimestamp: null,
			triggersPausedTimestamp: null,
			lastPutUpdateActiveAppletBinary: null,
			lastPutPrepareNextAppletTiming: null,
		});
	});

	yield takeEveryChannelMessage(appletTimingChannel, function* () {
		if (model.definitions !== null && model.definitions.length > 0 && !model.deviceInConnectedMode?.connected) {
			const newCurrentIndex = model.currentDefinitionIndex === null ? 0 : (model.currentDefinitionIndex + 1) % model.definitions.length;
			const newNextIndex = (newCurrentIndex + 1) % model.definitions.length;
			setModel({
				...model,
				currentDefinitionIndex: newCurrentIndex,
				startTimestamp: now().valueOf() as number,
				lastInteractionTimestamp: null,
				triggersPausedTimestamp: null,
			});
			const definition = model.definitions![newCurrentIndex!];
			yield* putUpdateActiveAppletBinaryIfChanged(definition);
			if (newNextIndex !== newCurrentIndex) {
				const nextDefinition = model.definitions![newNextIndex];
				yield* putPrepareNextAppletTimingIfChanged(nextDefinition);
			}
		}
	});

	yield takeEveryAndBindWhenPlatform(DeviceConnected, function* (action: IDeviceConnected) {
		setModel({
			...model,
			deviceInConnectedMode: {
				connected: true,
				appletUid: action.appletUid,
				appletVersion: action.appletVersion,
			},
		});
		yield putUpdateActiveAppletBinaryIfChanged(createConnectedAppletTimingDefinition(action.appletUid, action.appletVersion));
	});

	yield takeEveryAndBindWhenPlatform(DisconnectDevice, function* () {
		setModel({
			...model,
			deviceInConnectedMode: null,
		});
		yield put({
			type: PerformPowerAction,
			powerType: PowerActionType.APPLET_RELOAD,
		});
	});

	yield takeEvery(ActiveAppletRestore, function* () {
		if (model.deviceInConnectedMode?.connected) {
			yield putUpdateActiveAppletBinaryIfChanged(
				createConnectedAppletTimingDefinition(model.deviceInConnectedMode.appletUid, model.deviceInConnectedMode.appletVersion),
			);
		}
	});

	function* putUpdateActiveAppletBinaryIfChanged(definition: IAppletTimingDefinition) {
		const action: UpdateActiveAppletBinary = {
			type: UpdateActiveAppletBinary,
			activeAppletUid: definition.appletUid,
			activeConfiguration: definition.configuration,
			activeTimingChecksum: definition.timmingChecksum,
			activeAppletVersion: definition.appletVersion,
			activeAppletVersionPostfix: definition.appletVersionPostfix,
			activeAppletFrontAppletVersion: definition.frontAppletVersion,
			activeAppletIsPackage: definition.isPackage ?? false,
			activeAppletMetadata: definition.metadata,
		};
		if (!model.lastPutUpdateActiveAppletBinary || JSON.stringify(action) !== JSON.stringify(model.lastPutUpdateActiveAppletBinary)) {
			yield put(action);
			setModel({
				...model,
				lastPutUpdateActiveAppletBinary: action,
			});
		}
	}

	function* putPrepareNextAppletTimingIfChanged(nextDefinition: IAppletTimingDefinition) {
		const action: PrepareNextAppletTiming = {
			type: PrepareNextAppletTiming,
			appletUid: nextDefinition.appletUid,
			configuration: nextDefinition.configuration,
			timingChecksum: nextDefinition.timmingChecksum,
			appletVersion: nextDefinition.appletVersion,
			appletFrontAppletVersion: nextDefinition.frontAppletVersion,
			isPackage: nextDefinition.isPackage ?? false,
			appletMetadata: nextDefinition.metadata,
		};
		if (!model.lastPutPrepareNextAppletTiming || JSON.stringify(action) !== JSON.stringify(model.lastPutPrepareNextAppletTiming)) {
			yield put(action);
			setModel({
				...model,
				lastPutPrepareNextAppletTiming: action,
			});
		}
	}
}

export interface ILoopModel {
	definitions: IAppletTimingDefinition[] | null;
	currentDefinitionIndex: number | null;
	startTimestamp: number | null;
	lastInteractionTimestamp: number | null;
	triggersPausedTimestamp: number | null;
	lastPutUpdateActiveAppletBinary: UpdateActiveAppletBinary | null;
	lastPutPrepareNextAppletTiming: PrepareNextAppletTiming | null;
	deviceInConnectedMode: null | {
		connected: boolean;
		appletUid: string;
		appletVersion: string;
	};
}
