import { ILoopModel } from './appletTimingControllerSagas';
import { ShowNextAppletTiming } from './appletTimingActions';
import TimingFinishEventType from './TimingFinishEventType';
import { IAppletTimingDefinition } from '@signageos/actions/dist/Applet/appletActions';
import wait from '@signageos/lib/dist/Timer/wait';

const defaultModel: ILoopModel = {
	definitions: null,
	currentDefinitionIndex: null,
	startTimestamp: null,
	lastInteractionTimestamp: null,
	triggersPausedTimestamp: null,
	lastPutUpdateActiveAppletBinary: null,
	lastPutPrepareNextAppletTiming: null,
	deviceInConnectedMode: null,
};

class AppletTimingController {
	private model: ILoopModel;

	constructor(
		private getCurrentTimestamp: () => number,
		private postMessage: (message: ShowNextAppletTiming) => void,
	) {
		this.model = defaultModel;
	}

	public setModel(model: ILoopModel) {
		this.model = model;
	}

	public async run() {
		while (true) {
			try {
				const message = iterate(this.model, this.getCurrentTimestamp);
				if (message) {
					this.postMessage(message);
				}
			} catch (error) {
				// do nothing
			} finally {
				await wait(100);
			}
		}
	}
}

export function iterate(model: ILoopModel, getCurrentTimestamp: () => number): ShowNextAppletTiming | undefined {
	if (model.definitions === null || model.definitions.length === 0) {
		// nothing to show
		return;
	} else if (model.triggersPausedTimestamp !== null) {
		// don't do anything if it's paused and try again later
		return;
	} else if (model.currentDefinitionIndex === null) {
		return { type: ShowNextAppletTiming } as ShowNextAppletTiming;
	} else if (model.startTimestamp === null) {
		throw new Error('Unexpected state - start timestamp is null');
	} else {
		const currentDefinition = model.definitions[model.currentDefinitionIndex];
		if (showNext(currentDefinition, model.startTimestamp, model.lastInteractionTimestamp, getCurrentTimestamp())) {
			return { type: ShowNextAppletTiming } as ShowNextAppletTiming;
		}
	}
}

export function showNext(
	currentDefinition: IAppletTimingDefinition,
	startTimestamp: number,
	lastInteractionTimestamp: number | null,
	currentTimestamp: number,
) {
	switch (TimingFinishEventType[currentDefinition.finishEvent.type as keyof typeof TimingFinishEventType]) {
		case TimingFinishEventType.DURATION:
			return startTimestamp + parseFinishEventData(currentDefinition) <= currentTimestamp;
		case TimingFinishEventType.IDLE_TIMEOUT:
			if (lastInteractionTimestamp !== null && lastInteractionTimestamp > startTimestamp) {
				return lastInteractionTimestamp + parseFinishEventData(currentDefinition) <= currentTimestamp;
			}
			return startTimestamp + parseFinishEventData(currentDefinition) <= currentTimestamp;
		case TimingFinishEventType.SCREEN_TAP:
			return lastInteractionTimestamp !== null && lastInteractionTimestamp <= currentTimestamp;
		default:
			throw new Error('Invalid timing finish event type: ' + currentDefinition.finishEvent.type);
	}
}

function parseFinishEventData(definition: IAppletTimingDefinition) {
	switch (TimingFinishEventType[definition.finishEvent.type as keyof typeof TimingFinishEventType]) {
		case TimingFinishEventType.DURATION:
			return definition.finishEvent.data !== null ? parseInt(definition.finishEvent.data) : 60e3;
		case TimingFinishEventType.IDLE_TIMEOUT:
			return definition.finishEvent.data !== null ? parseInt(definition.finishEvent.data) : 30e3;
		case TimingFinishEventType.SCREEN_TAP:
			throw new Error('Unsupported event data for type: ' + definition.finishEvent.type);
		default:
			throw new Error('Unrecognized type of finishEvent: ' + definition.finishEvent.type);
	}
}

export default AppletTimingController;
