import { SagaIterator } from '@redux-saga/types';
import { ActiveAppletLoaded, ActiveAppletReady } from '@signageos/actions/dist/Applet/activeAppletActions';
import { UpdateActiveAppletBinary, UpdateActiveAppletUniqueHash } from '@signageos/actions/dist/Applet/appletActions';
import { UpdateAuthentication } from '@signageos/actions/dist/Device/Authentication/deviceAuthenticationActions';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import Debug from 'debug';
import { buffers, channel } from 'redux-saga';
import { call, fork, put, takeEvery } from 'redux-saga/effects';
import { withDependencies } from '../../DI/dependencyInjection';
import { FrontCacheDriver, normalizeFrontCacheDriver } from '../../NativeDevice/Default/combinedDriver';
import IManagementDriver, { IFrontManagementDriver } from '../../NativeDevice/Management/IManagementDriver';
import OfflineCache from '../../OfflineCache/OfflineCache';
import { bindAndTakeEvery, takeEveryAndBindWhenPlatform } from '../../Socket/socketActionCreator';
import { IPolymorphicSynchronizer } from '../../Synchronization/PolymorphicSynchronizer/IPolymorphicSynchronizer';
import { IProprietaryTimerStorage } from '../../Timer/ITimerStorage';
import { CreateAppletIframe, HideOSD, RemoveAppletIframe, StartAppletIframe } from '../AppletTiming/appletTimingActions';
import { IFrontState } from '../frontReducers';
import { createErrorTransferObject } from './Error/errorHelper';
import IMessage from './IMessage';
import sendAppletMessage, { sendMessageToActiveAppletIfExists } from './sendAppletMessage';
import * as appletMessageHandler from './appletMessageHandler';
import { HandlerResult } from './IHandler';
import ManagementCapabilities from '../../NativeDevice/Management/ManagementCapability';
import { IPropertyStorage } from '../../Property/propertyStorage';
import { MonitoringLogData } from '@signageos/common-types/dist/Device/MonitoringLog/MonitoringLogData';
import { DeviceTelemetryType } from '@signageos/common-types/dist/Device/Telemetry/DeviceTelemetryType';
import Property from '../../Property/Property';
import { UpdateDeviceTelemetryRecord } from '@signageos/actions/dist/Device/Telemetry/deviceTelemetryActions';
import { isEqual } from 'lodash';
import { deliver } from '../../Socket/socketActionDeliverHelper';

const debug = Debug('@signageos/front-display:Front:Applet:appletSagas');

export function* updateAppletUniqueHash() {
	let activeAppletUniqueHash: string | null = null;
	let lastUpdateActiveAppletBinary: UpdateActiveAppletBinary | null = null;

	yield takeEveryAndBindWhenPlatform(
		UpdateActiveAppletUniqueHash,
		function* (updateActiveAppletUniqueHash: UpdateActiveAppletUniqueHash): IterableIterator<never> {
			activeAppletUniqueHash = updateActiveAppletUniqueHash.hash;
		},
	);
	yield takeEvery(HideOSD, function* (): IterableIterator<never> {
		lastUpdateActiveAppletBinary = null;
	});
	yield takeEveryAndBindWhenPlatform(UpdateActiveAppletBinary, function* (updateActiveAppletBinary: UpdateActiveAppletBinary) {
		try {
			const changed =
				!lastUpdateActiveAppletBinary ||
				lastUpdateActiveAppletBinary.activeAppletUid !== updateActiveAppletBinary.activeAppletUid ||
				lastUpdateActiveAppletBinary.activeAppletVersion !== updateActiveAppletBinary.activeAppletVersion ||
				JSON.stringify(lastUpdateActiveAppletBinary.activeConfiguration) !== JSON.stringify(updateActiveAppletBinary.activeConfiguration);
			if (changed || activeAppletUniqueHash === null) {
				yield put({
					type: UpdateActiveAppletUniqueHash,
					hash: generateUniqueHash(),
				} as UpdateActiveAppletUniqueHash);
			}
			lastUpdateActiveAppletBinary = updateActiveAppletBinary;
		} catch (error) {
			console.error('updateAppletUniqueHash failed', error);
		}
	});
}

export function* updateAppletDeviceAuthHash(messageTypePrefix: string, window: Window, getState: () => IFrontState) {
	yield bindAndTakeEvery(UpdateAuthentication, function* (action: UpdateAuthentication) {
		const deviceAuthHash = action.authHash;
		yield sendMessageToActiveAppletIfExists(window, getState, {
			type: messageTypePrefix + '.auth_hash.update',
			authHash: deviceAuthHash,
		});
	});
}

export function* bindAppletMessages(
	messageTypePrefix: string,
	window: Window,
	getState: () => IFrontState,
	getNativeDriver: () => FrontCacheDriver,
	getSynchronizer: () => IPolymorphicSynchronizer,
	offlineCache: OfflineCache,
	shortAppletFilesUrl: boolean,
	timerStorage: IProprietaryTimerStorage,
) {
	const messageChannel = createMessageEventChannel(window);
	function* callback(managementDriver: IManagementDriver, applicationVersion: string, event: MessageEvent<any>): SagaIterator {
		try {
			const data: IMessage = event.data;
			const appletState = getState().applet;

			let appletUid: string;
			let iframeId: string;
			let timingChecksum: string;

			const activeAppletIframe =
				appletState.activeAppletIframeId && (window.document.getElementById(appletState.activeAppletIframeId) as HTMLIFrameElement);
			const nextAppletIframe =
				appletState.nextApplet &&
				appletState.nextApplet.iframeId &&
				(window.document.getElementById(appletState.nextApplet.iframeId) as HTMLIFrameElement);

			if (activeAppletIframe && event.source === activeAppletIframe.contentWindow) {
				appletUid = appletState.activeAppletUid!;
				iframeId = appletState.activeAppletIframeId!;
				timingChecksum = appletState.activeTimingChecksum!;
			} else if (nextAppletIframe && event.source === nextAppletIframe.contentWindow) {
				appletUid = appletState.nextApplet!.uid;
				iframeId = appletState.nextApplet!.iframeId!;
				timingChecksum = appletState.nextApplet!.timingChecksum;
			} else {
				// Silent not continuing on unknown source window
				return;
			}

			if (!appletUid) {
				throw new Error('Applet UID is not set');
			}
			if (!timingChecksum) {
				throw new Error('Timing checksum is not set');
			}
			const invocationUid = data.invocationUid;
			try {
				debug('handle message from applet: ' + data.type, data);
				const response = yield call(
					handleAppletMessage,
					messageTypePrefix,
					data,
					appletUid,
					timingChecksum,
					getNativeDriver,
					() => managementDriver,
					getSynchronizer,
					offlineCache,
					window,
					getState,
					applicationVersion,
					shortAppletFilesUrl,
					timerStorage,
				);
				debug('message OK: ' + data.type, data);
				yield call(sendAppletMessage, window, iframeId, {
					type: messageTypePrefix + '.invocation.success',
					invocationUid,
					...response,
				});
			} catch (error) {
				debug('message FAILED: ' + data.type, data);
				yield call(sendAppletMessage, window, iframeId, {
					type: messageTypePrefix + '.invocation.error',
					invocationUid,
					error: createErrorTransferObject(error),
				});
			}
		} catch (error) {
			debug('bindAppletMassages failed', error);
		}
	}
	yield fork(
		withDependencies(['managementDriver', 'applicationVersion'], function* ({ managementDriver, applicationVersion }) {
			yield takeEvery(messageChannel, callback, managementDriver, applicationVersion);
		}),
	);
}

function* handleAppletMessage(
	messageTypePrefix: string,
	data: IMessage,
	appletUid: string,
	timingChecksum: string,
	getNativeDriver: () => FrontCacheDriver,
	getManagementDriver: () => IManagementDriver,
	getSynchronizer: () => IPolymorphicSynchronizer,
	offlineCache: OfflineCache,
	window: Window,
	getState: () => IFrontState,
	applicationVersion: string,
	shortAppletFilesUrl: boolean,
	timerStorage: IProprietaryTimerStorage,
): HandlerResult {
	debug('Handling applet message', data, appletUid, timingChecksum);

	const { frontDriver, cacheDriver } = normalizeFrontCacheDriver(getNativeDriver());
	return yield appletMessageHandler.handleAppletMessage({
		data,
		window,
		appletUid,
		offlineCache,
		timingChecksum,
		messageTypePrefix,
		synchronizer: getSynchronizer(),
		frontDriver,
		cacheDriver,
		managementDriver: getManagementDriver(),
		getState,
		applicationVersion,
		shortAppletFilesUrl,
		timerStorage,
	});
}

export function* bindAppletIframe(
	messageTypePrefix: string,
	window: Window,
	getState: () => IFrontState,
	frontDisplayVersion: string,
	getNativeDriver: () => IFrontManagementDriver,
) {
	const iframeReadyPromises: { [id: string]: Promise<any> | undefined } = {};

	yield takeEvery(CreateAppletIframe, function* (action: CreateAppletIframe): IterableIterator<any> {
		iframeReadyPromises[action.id] = initializeApplet(
			window,
			action.uid,
			action.id,
			messageTypePrefix,
			getState,
			frontDisplayVersion,
			getNativeDriver,
		);
	});
	yield takeEvery(RemoveAppletIframe, function* (action: RemoveAppletIframe): IterableIterator<any> {
		delete iframeReadyPromises[action.id];
	});
	yield takeEvery(StartAppletIframe, function* (action: StartAppletIframe) {
		try {
			const state = getState();
			const appletIframeId = action.id;
			const appletUid = action.uid;

			if (iframeReadyPromises[appletIframeId]) {
				yield iframeReadyPromises[appletIframeId];
			} else {
				yield initializeApplet(window, appletUid, appletIframeId, messageTypePrefix, getState, frontDisplayVersion, getNativeDriver);
			}
			const timingChecksum = state.applet.activeTimingChecksum ?? '';
			yield put<ActiveAppletLoaded>({ type: ActiveAppletLoaded, appletUid, timingChecksum });

			if (state.deviceConnect.connectedBaseUrl) {
				yield sendAppletMessage(window, appletIframeId, {
					type: messageTypePrefix + '.connect.update',
					connectedBaseUrl: state.deviceConnect.connectedBaseUrl,
				});
			}

			const deviceAuthHash = state.deviceAuthentication.authHash;
			yield sendAppletMessage(window, appletIframeId, {
				type: messageTypePrefix + '.auth_hash.update',
				authHash: deviceAuthHash,
			});
			yield put<ActiveAppletReady>({ type: ActiveAppletReady, appletUid, timingChecksum });
		} catch (error) {
			debug('bindAppletIframe failed', error);
		}
	});
}

async function initializeApplet(
	window: Window,
	appletUid: string,
	iframeId: string,
	messageTypePrefix: string,
	getState: () => IFrontState,
	frontDisplayVersion: string,
	managementDriver: () => IFrontManagementDriver,
) {
	const state = getState();
	let appletConfiguration = state.applet.activeAppletConfiguration;
	let frontAppletJsFile = state.applet.activeAppletFrontAppletJsFile;
	let hasBundledFrontApplet = state.applet.activeAppletHasBundledFrontApplet;
	let appletVersion = state.applet.activeAppletVersion;
	if (state.applet.nextApplet && appletUid === state.applet.nextApplet.uid) {
		appletConfiguration = state.applet.nextApplet.configuration;
		frontAppletJsFile = state.applet.nextApplet.appletFrontAppletJsFile;
		hasBundledFrontApplet = state.applet.nextApplet.hasBundledFrontApplet;
		appletVersion = state.applet.nextApplet.appletVersion;
	}

	if (!hasBundledFrontApplet) {
		await waitForAppletMessage(window, iframeId, messageTypePrefix + '_loader.ready');
		if (!frontAppletJsFile) {
			throw new Error('Front applet js file was not loaded yet');
		}
		await sendAppletMessage(window, iframeId, {
			type: messageTypePrefix + '.api_js_uri',
			uri: frontAppletJsFile.localUri,
		});
	}
	await waitForAppletMessage(window, iframeId, messageTypePrefix + '_api.ready');
	await sendAppletMessage(window, iframeId, {
		type: messageTypePrefix + '.front_display_version.update',
		version: frontDisplayVersion,
	});
	await sendAppletMessage(window, iframeId, {
		type: messageTypePrefix + '.applet_version.update',
		version: appletVersion,
	});

	const newConfig = await handleDecryptedFields(state, appletConfiguration, managementDriver);

	await sendAppletMessage(window, iframeId, {
		type: messageTypePrefix + '.config.update',
		config: newConfig,
	});
}

function createMessageEventChannel(window: Window) {
	const messageChannel = channel<MessageEvent>(buffers.dropping(10));
	window.addEventListener('message', (event: MessageEvent) => {
		messageChannel.put(event);
	});
	return messageChannel;
}

function waitForAppletMessage(window: Window, appletIframeId: string, type: string) {
	const appletIframe = window.document.getElementById(appletIframeId) as HTMLIFrameElement;
	if (!appletIframe) {
		throw new Error('Not found active applet iframe by id ' + appletIframeId);
	}
	if (!appletIframe.contentWindow) {
		throw new Error('Not found active applet iframe contentWindow ' + appletIframeId);
	}
	const appletWindow = appletIframe.contentWindow.window;
	return new Promise<void>((resolve: () => void) => {
		// The global MessageEvent type conflicts with service worker MessageEvent type. This is only a workaround.
		type WindowMessageEvent = Parameters<Parameters<typeof window.addEventListener<'message'>>[1]>[0];
		const messageListener = (event: WindowMessageEvent) => {
			const sourceWindow = event.source;
			if (sourceWindow === appletWindow) {
				const data = event.data;
				switch (data.type) {
					case type:
						window.removeEventListener('message', messageListener);
						resolve();
						break;
					default:
				}
			}
		};
		window.addEventListener('message', messageListener);
	});
}

async function handleDecryptedFields(
	state: IFrontState,
	appletConfiguration: Record<string, any>,
	managementDriver: () => IFrontManagementDriver,
): Promise<Record<string, any>> {
	const getEncryptedFields = state.applet.activeAppletMetadata?.encryptedConfig || {};
	const encryptedKeys = Object.keys(getEncryptedFields);
	const isSupported = await managementDriver().managementSupports(ManagementCapabilities.SECRETS);
	try {
		if (encryptedKeys.length > 0 && isSupported) {
			const newConfig = { ...appletConfiguration };

			for (const key of encryptedKeys) {
				const valueFromConfiguration = state.applet.activeAppletConfiguration[key];
				if (valueFromConfiguration) {
					const jweGeneral = JSON.parse(Buffer.from(valueFromConfiguration, 'base64').toString('utf-8'));
					const decryptedValue = await managementDriver().secretManager.decryptJweGeneralToUtf8(jweGeneral);
					newConfig[key] = decryptedValue;
				}
			}

			return newConfig;
		}
	} catch (error) {
		console.error('Failed to decrypt applet configuration', error);
	}

	return appletConfiguration;
}

export function* notifyAppletTelemetry(propertyStorage: IPropertyStorage) {
	yield takeEvery(UpdateActiveAppletBinary, function* (action: UpdateActiveAppletBinary): IterableIterator<any> {
		try {
			if (action.activeAppletUid && action.activeAppletUid && action.activeAppletVersion) {
				const currentAppletSetting: MonitoringLogData[DeviceTelemetryType.APPLET] = {
					appletUid: action.activeAppletUid,
					appletVersion: action.activeAppletVersion,
					config: action.activeConfiguration,
				};

				const lastUpdated: MonitoringLogData[DeviceTelemetryType.APPLET] | undefined = yield propertyStorage.getValueOrDefault(
					Property.LATEST_REPORTED_SETTINGS_APPLET,
					{},
				);

				if (!isEqual(lastUpdated, currentAppletSetting)) {
					yield deliver<UpdateDeviceTelemetryRecord<DeviceTelemetryType.APPLET, MonitoringLogData[DeviceTelemetryType.APPLET]>>({
						type: UpdateDeviceTelemetryRecord,
						name: DeviceTelemetryType.APPLET,
						data: currentAppletSetting,
					});
					yield propertyStorage.setValue(Property.LATEST_REPORTED_SETTINGS_APPLET, currentAppletSetting);
				}
			}
		} catch (error) {
			console.error('notifyAppletTelemetry failed', error);
		}
	});
}
