import { IFile as IFileSystemFile } from '../../NativeDevice/fileSystem';
import {
	UpdateActiveAppletBinary,
	SubscribeActiveApplet,
	UpdateActiveAppletUniqueHash,
	UpdateActiveAppletBinaryFile,
	UpdateAppletTimingsDefinition,
} from '@signageos/actions/dist/Applet/appletActions';
import {
	CreateAppletIframe,
	StartAppletIframe,
	PrepareNextAppletTiming,
	UpdateNextAppletBinaryFile,
	ShowOSD,
	HideApplet,
	ShowOSDTime,
} from '../AppletTiming/appletTimingActions';
import { TriggerDisplayPowerOn, TriggerDisplayPowerOff, TriggerAppletOn, TriggerAppletOff } from '../Device/Power/devicePowerActionActions';
import { ChangeLicenseState, ViolationSeverity } from '../License/licenseActions';

export interface IAppletState {
	isCachedAppletReady: boolean;
	activeAppletUid: string | null;
	activeAppletConfiguration: any | null;
	activeAppletMissing: boolean;
	activeAppletUniqueHash: string | null;
	activeAppletIframeId: string | null;
	activeAppletVersion: string | null;
	activeTimingChecksum: string | null;
	activeAppletBinaryFile: IFileSystemFile | null;
	activeAppletFrontAppletJsFile: IFileSystemFile | null;
	activeAppletHasBundledFrontApplet: boolean;
	appletIsMuted: boolean;
	nextApplet: null | {
		uid: string;
		configuration: any;
		timingChecksum: string;
		appletVersion: string;
		appletFrontAppletVersion: string | null;
		appletBinaryFile: IFileSystemFile | null;
		appletFrontAppletJsFile: IFileSystemFile | null;
		hasBundledFrontApplet: boolean;
		iframeId: string | null;
		metadata: any;
	};
	activeAppletMetadata?: {
		encryptedConfig?: Record<string, string>;
	};
	lastOpenedOSDTime: number;
}

const initialState: IAppletState = {
	isCachedAppletReady: false,
	activeAppletUid: null,
	activeAppletConfiguration: null,
	activeAppletMissing: false,
	activeAppletUniqueHash: null,
	activeAppletIframeId: null,
	activeAppletVersion: null,
	activeTimingChecksum: null,
	activeAppletBinaryFile: null,
	activeAppletFrontAppletJsFile: null,
	activeAppletHasBundledFrontApplet: false,
	appletIsMuted: false,
	nextApplet: null,
	activeAppletMetadata: {},
	lastOpenedOSDTime: 0,
};

const OSD_UID = 'osd';

export function applet(
	state: IAppletState = initialState,
	action:
		| TriggerDisplayPowerOn
		| TriggerDisplayPowerOff
		| UpdateActiveAppletBinary
		| UpdateActiveAppletUniqueHash
		| UpdateAppletTimingsDefinition
		| CreateAppletIframe
		| StartAppletIframe
		| UpdateActiveAppletBinaryFile
		| PrepareNextAppletTiming
		| UpdateNextAppletBinaryFile
		| TriggerAppletOn
		| TriggerAppletOff
		| ChangeLicenseState
		| ShowOSD
		| ShowOSDTime
		| HideApplet
		| SubscribeActiveApplet,
): IAppletState {
	switch (action.type) {
		case UpdateActiveAppletBinary:
			const changed =
				state.activeAppletUid !== action.activeAppletUid ||
				state.activeAppletVersion !== action.activeAppletVersion ||
				JSON.stringify(state.activeAppletConfiguration) !== JSON.stringify(action.activeConfiguration);
			return {
				...state,
				activeAppletUid: action.activeAppletUid,
				activeAppletConfiguration: action.activeConfiguration,
				activeTimingChecksum: action.activeTimingChecksum,
				activeAppletVersion: action.activeAppletVersion,
				activeAppletMissing: false,
				activeAppletUniqueHash: changed ? null : state.activeAppletUniqueHash,
				activeAppletIframeId: changed ? null : state.activeAppletIframeId,
				activeAppletBinaryFile: changed ? null : state.activeAppletBinaryFile,
				activeAppletFrontAppletJsFile: changed ? null : state.activeAppletFrontAppletJsFile,
				activeAppletHasBundledFrontApplet: changed ? !action.activeAppletFrontAppletVersion : state.activeAppletHasBundledFrontApplet,
				nextApplet: null,
				activeAppletMetadata: action.activeAppletMetadata,
			};
		case ShowOSD:
			return {
				...state,
				activeAppletUid: OSD_UID,
				activeAppletConfiguration: {},
				activeTimingChecksum: OSD_UID,
				activeAppletVersion: null,
				activeAppletMissing: false,
				activeAppletUniqueHash: OSD_UID,
				activeAppletIframeId: null,
				activeAppletBinaryFile: action.appletBinaryFile,
				activeAppletFrontAppletJsFile: null,
				activeAppletHasBundledFrontApplet: true,
				nextApplet: null,
			};
		case ShowOSDTime:
			return {
				...state,
				lastOpenedOSDTime: action.time,
			};
		case HideApplet:
			return {
				...state,
				activeAppletMissing: true,
			};
		case SubscribeActiveApplet:
			return {
				...state,
				activeAppletMissing: false,
			};
		case UpdateActiveAppletUniqueHash:
			return {
				...state,
				activeAppletUniqueHash: action.hash,
			};
		case CreateAppletIframe:
			const createAppletIframeUid = action.uid;
			const createAppletIframeId = action.id;
			if (state.activeAppletUid === createAppletIframeUid) {
				return {
					...state,
					activeAppletIframeId: createAppletIframeId,
				};
			} else if (state.nextApplet && state.nextApplet.uid === createAppletIframeUid) {
				return {
					...state,
					nextApplet: {
						...state.nextApplet,
						iframeId: createAppletIframeId,
					},
				};
			}

			return state;
		case StartAppletIframe:
			return {
				...state,
				activeAppletIframeId: action.id,
			};
		case UpdateActiveAppletBinaryFile:
			return {
				...state,
				activeAppletBinaryFile: action.appletBinaryFile,
				activeAppletFrontAppletJsFile: action.appletFrontAppletJsFile,
			};

		case TriggerAppletOn:
			return {
				...state,
				appletIsMuted: false,
			};

		case TriggerAppletOff:
			return {
				...state,
				appletIsMuted: true,
			};

		case ChangeLicenseState:
			return {
				...state,
				appletIsMuted: action.severity !== ViolationSeverity.none,
			};

		case PrepareNextAppletTiming:
			return {
				...state,
				nextApplet: {
					uid: action.appletUid,
					configuration: action.configuration,
					timingChecksum: action.timingChecksum,
					appletVersion: action.appletVersion,
					appletFrontAppletVersion: action.appletFrontAppletVersion || null,
					appletBinaryFile: null,
					appletFrontAppletJsFile: null,
					hasBundledFrontApplet: !!action.appletFrontAppletVersion,
					iframeId: null,
					metadata: action.appletMetadata,
				},
			};
		case UpdateNextAppletBinaryFile:
			if (state.nextApplet && state.nextApplet.uid === action.appletUid) {
				return {
					...state,
					nextApplet: {
						...state.nextApplet,
						appletBinaryFile: action.appletBinaryFile,
						appletFrontAppletJsFile: action.appletFrontAppletJsFile,
					},
				};
			} else {
				return state;
			}
		case UpdateAppletTimingsDefinition:
			return {
				...state,
				isCachedAppletReady: action.definitions.length > 0,
				activeAppletMetadata: action.definitions[0].metadata,
			};
		default:
			return state;
	}
}
