import { bindAndTakeEvery, takeEveryAndBindWhenPlatform } from '../../Socket/socketActionCreator';
import { fork, put, takeEvery } from 'redux-saga/effects';
import { IFrontState } from '../frontReducers';
import IDriver from '../../NativeDevice/Front/IFrontDriver';
import { AuthenticationSucceed } from '@signageos/actions/dist/Authentication/authenticationActions';
import { VerificationSucceeded } from '@signageos/actions/dist/Device/Verification/deviceVerificationActions';
import {
	SubscribeActiveApplet,
	UpdateActiveAppletBinary,
	UpdateActiveAppletBinaryFile,
} from '@signageos/actions/dist/Applet/appletActions';
import OfflineCache from '../../OfflineCache/OfflineCache';
import { PrepareNextAppletTiming, UpdateNextAppletBinaryFile } from '../AppletTiming/appletTimingActions';
import { AppletBinaryDownloader } from './AppletBinaryDownloader';
import { awaitDependencies, withDependencies } from '../../DI/dependencyInjection';

export function* activeAppletBinary(getState: () => IFrontState, getNativeDriver: () => IDriver) {
	yield bindAndTakeEvery([AuthenticationSucceed, VerificationSucceeded], function* (action: AuthenticationSucceed | VerificationSucceeded) {
		try {
			const deviceUid: string = yield getNativeDriver().getDeviceUid();

			switch (action.type) {
				case AuthenticationSucceed:
					const state = getState();
					if (state.deviceVerification.verified) {
						yield put({
							type: SubscribeActiveApplet,
							deviceUid,
						} as SubscribeActiveApplet);
					}
					break;
				case VerificationSucceeded:
					yield put({
						type: SubscribeActiveApplet,
						deviceUid,
					} as SubscribeActiveApplet);
					break;
				default:
			}
		} catch (error) {
			console.error('activeAppletBinary failed', error);
		}
	});
}

export function* updateActiveAppletBinaryFile(
	shortAppletFilesUrl: boolean,
	offlineCache: OfflineCache,
	defaultTimingData?: {
		appletUid: string;
		appletVersion: string;
		appletBinaryFile: string;
		appletFrontAppletJsFile: string | undefined;
	},
) {
	let staticBaseUrl: string | null = null;
	yield fork(function* () {
		const { staticBaseUrl: staticBaseUrlFromState } = yield awaitDependencies(['staticBaseUrl']);
		staticBaseUrl = staticBaseUrlFromState;
	});
	yield fork(
		withDependencies(['connectedBaseUrl'], function* ({ connectedBaseUrl }) {
			const getStaticBaseUrl = () => {
				const finalStaticBaseUrl = connectedBaseUrl ?? staticBaseUrl;
				if (finalStaticBaseUrl === null) {
					throw new Error(`The applet is not downloaded and is trying to download before the staticBaseUrl is set.`);
				}
				return finalStaticBaseUrl;
			};
			const appletBinaryDownloader = new AppletBinaryDownloader(shortAppletFilesUrl, offlineCache, getStaticBaseUrl);

			yield takeEveryAndBindWhenPlatform(UpdateActiveAppletBinary, function* (updateActiveAppletBinary: UpdateActiveAppletBinary) {
				try {
					type Bins = Awaited<ReturnType<typeof appletBinaryDownloader.getOrDownloadAppletBinaries>>;

					const { frontAppletJsFile, appletBinaryFile }: Bins = yield appletBinaryDownloader.getOrDownloadAppletBinaries(
						updateActiveAppletBinary.activeAppletUid,
						updateActiveAppletBinary.activeAppletVersion,
						updateActiveAppletBinary.activeAppletIsPackage ?? false,
						updateActiveAppletBinary.activeAppletFrontAppletVersion || updateActiveAppletBinary.activeAppletHugApiVersion!, // deprecated will be deleted
						updateActiveAppletBinary.activeAppletVersionPostfix,
						defaultTimingData,
					);
					yield put({
						type: UpdateActiveAppletBinaryFile,
						appletBinaryFile,
						appletFrontAppletJsFile: frontAppletJsFile,
					} as UpdateActiveAppletBinaryFile);
				} catch (error) {
					console.error('updateActiveAppletBinaryFile failed', error);
				}
			});
			yield takeEvery(PrepareNextAppletTiming, function* (action: PrepareNextAppletTiming) {
				try {
					const { frontAppletJsFile, appletBinaryFile } = yield appletBinaryDownloader.getOrDownloadAppletBinaries(
						action.appletUid,
						action.appletVersion,
						action.isPackage,
						action.appletFrontAppletVersion,
						action.appletVersionPostfix,
						defaultTimingData,
					);
					yield put({
						type: UpdateNextAppletBinaryFile,
						appletUid: action.appletUid,
						appletBinaryFile,
						appletFrontAppletJsFile: frontAppletJsFile,
					} as UpdateNextAppletBinaryFile);
				} catch (error) {
					console.error('prepareNextAppletTiming failed', error);
				}
			});
		}),
	);
}
