import { put, spawn, take } from 'redux-saga/effects';
import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import INativeVideoEvent from '../../../Video/IVideoEvent';
import IVideoPlayMessage from './IVideoPlayMessage';
import IVideoPrepareMessage from './IVideoPrepareMessage';
import IVideoStopMessage from './IVideoStopMessage';
import IVideoPauseMessage from './IVideoPauseMessage';
import IVideoResumeMessage from './IVideoResumeMessage';
import { buffers, eventChannel } from 'redux-saga';
import {
	ActiveAppletVideoPlay,
	ActiveAppletVideoPrepare,
	ActiveAppletVideoStop,
	ActiveAppletVideoPause,
	ActiveAppletVideoResume,
} from './appletVideoActions';
import { EmitVideoEvent } from './EmitVideoEvent';
import { ActiveAppletRestore } from '../activeAppletActions';
import IVideoEventEmitter from '../../../Video/IVideoEventEmitter';
import InternalVideoError from '../Error/InternalVideoError';
import ErrorCodes from '../Error/ErrorCodes';
import IVideoMessage from './IVideoMessage';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { NoMoreAvailableVideosError } from '../../../NativeDevice/Error/videoErrors';

export function* handleVideoMessage(
	messageTypePrefix: string,
	data: IVideoPrepareMessage | IVideoPlayMessage | IVideoStopMessage | IVideoPauseMessage | IVideoResumeMessage,
	nativeDriver: IDriver,
	appletUid: string,
	timingChecksum: string,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.restore':
			return yield handleRestore();
		case messageTypePrefix + '.video.play':
			return yield handleVideoPlay(messageTypePrefix, data, nativeDriver, appletUid, timingChecksum);
		case messageTypePrefix + '.video.prepare':
			return yield handleVideoPrepare(data, nativeDriver, appletUid, timingChecksum);
		case messageTypePrefix + '.video.stop':
			return yield handleVideoStop(data, nativeDriver, appletUid, timingChecksum);
		case messageTypePrefix + '.video.pause':
			return yield handleVideoPause(data, nativeDriver, appletUid, timingChecksum);
		case messageTypePrefix + '.video.resume':
			return yield handleVideoResume(data, nativeDriver, appletUid, timingChecksum);
		default:
			return null;
	}
}

function* handleRestore() {
	yield put({ type: ActiveAppletRestore } as ActiveAppletRestore);
	return {};
}

function* handleVideoPlay(
	messageTypePrefix: string,
	data: IVideoPlayMessage,
	nativeDriver: IDriver,
	appletUid: string,
	timingChecksum: string,
): HandlerResult {
	let nativeVideo: IVideoEventEmitter;

	try {
		nativeVideo = (yield nativeDriver.video.play(data.uri, data.x, data.y, data.width, data.height)) as any;
	} catch (error) {
		throw new InternalVideoError({
			kind: 'internalVideoErrorWithOrigin',
			message: "Couldn't play the video.",
			code: ErrorCodes.VIDEO_PLAY_ERROR,
			origin: `play(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	const nativeVideoEndedChannel = eventChannel<{ type: string; event: INativeVideoEvent }>(
		(putEvent: (message: { type: string; event: INativeVideoEvent }) => void) => {
			const onEnded = (event: INativeVideoEvent) => putEvent({ type: 'ended', event });
			const onError = (event: INativeVideoEvent) => putEvent({ type: 'error', event });
			const onStopped = (event: INativeVideoEvent) => putEvent({ type: 'stopped', event });
			nativeVideo.once('ended', onEnded);
			nativeVideo.once('error', onError);
			nativeVideo.once('stopped', onStopped);
			return () => {
				nativeVideo.removeListener('ended', onEnded);
				nativeVideo.removeListener('error', onError);
				nativeVideo.removeListener('stopped', onStopped);
			};
		},
		buffers.dropping(1),
	);
	yield spawn(function* () {
		const message: { type: string; event: INativeVideoEvent } = yield take(nativeVideoEndedChannel);
		try {
			const event = message.event;
			yield put({
				type: EmitVideoEvent,
				event: {
					type: message.type,
					srcArguments: event.srcArguments,
					data: event.data,
				},
			} as EmitVideoEvent);
		} catch (error) {
			console.error(messageTypePrefix + '.video.play event sending failed', error);
		} finally {
			nativeVideoEndedChannel.close();
		}
	});
	yield put<ActiveAppletVideoPlay>({ type: ActiveAppletVideoPlay, appletUid, timingChecksum, data });
	return {};
}

function* handleVideoPrepare(data: IVideoPrepareMessage, nativeDriver: IDriver, appletUid: string, timingChecksum: string) {
	if (data.options && typeof data.options.volume !== 'undefined' && (data.options.volume < 0 || data.options.volume > 100)) {
		throw new Error('Volume out of bounds: ' + data.options.volume);
	}

	try {
		yield nativeDriver.video.prepare(data.uri, data.x, data.y, data.width, data.height, data.options);
	} catch (error) {
		if (error instanceof NoMoreAvailableVideosError) {
			console.warn('video prepare failed because there are no more available video players');
		} else {
			throw new InternalVideoError({
				kind: 'internalVideoErrorWithOrigin',
				message: "Couldn't prepare the video.",
				code: ErrorCodes.VIDEO_PREPARE_ERROR,
				origin: `prepare(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
				originStack: error.stack,
				originMessage: error.message,
			});
		}
	}

	yield put<ActiveAppletVideoPrepare>({ type: ActiveAppletVideoPrepare, appletUid, timingChecksum, data });
	return {};
}
function* handleVideoStop(data: IVideoStopMessage, nativeDriver: IDriver, appletUid: string, timingChecksum: string) {
	try {
		yield nativeDriver.video.stop(data.uri, data.x, data.y, data.width, data.height);
	} catch (error) {
		throw new InternalVideoError({
			kind: 'internalVideoErrorWithOrigin',
			message: "Couldn't stop the video.",
			code: ErrorCodes.VIDEO_STOP_ERROR,
			origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	yield put<ActiveAppletVideoStop>({ type: ActiveAppletVideoStop, appletUid, timingChecksum, data });
	return {};
}

function* handleVideoPause(data: IVideoPauseMessage, nativeDriver: IDriver, appletUid: string, timingChecksum: string) {
	try {
		yield nativeDriver.video.pause(data.uri, data.x, data.y, data.width, data.height);
	} catch (error) {
		throw new InternalVideoError({
			kind: 'internalVideoErrorWithOrigin',
			message: "Couldn't pause the video.",
			code: ErrorCodes.VIDEO_PAUSE_ERROR,
			origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	yield put<ActiveAppletVideoPause>({ type: ActiveAppletVideoPause, appletUid, timingChecksum, data });
	return {};
}

function* handleVideoResume(data: IVideoResumeMessage, nativeDriver: IDriver, appletUid: string, timingChecksum: string) {
	try {
		yield nativeDriver.video.resume(data.uri, data.x, data.y, data.width, data.height);
	} catch (error) {
		throw new InternalVideoError({
			kind: 'internalVideoErrorWithOrigin',
			message: "Couldn't resume the video.",
			code: ErrorCodes.VIDEO_RESUME_ERROR,
			origin: `stop(${data.uri}, ${data.x}, ${data.y}, ${data.width}, ${data.height})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}

	yield put<ActiveAppletVideoResume>({ type: ActiveAppletVideoResume, appletUid, timingChecksum, data });
	return {};
}

export default function* videoHandler({ messageTypePrefix, data, frontDriver, appletUid, timingChecksum }: IHandlerParams): HandlerResult {
	return yield handleVideoMessage(messageTypePrefix, data as IVideoMessage, frontDriver, appletUid, timingChecksum);
}
