import Debug from 'debug';
import { spawn } from 'redux-saga/effects';
import { IFrontState } from '../../frontReducers';
import { createChannel, takeEvery as takeEveryChannelMessage } from '../../../ReduxSaga/channels';
import { sendMessageToActiveAppletIfExists } from '../sendAppletMessage';
import ISynchronizer, { SynchronizerEvent, BroadcastedValue, GroupStatus } from '../../../Synchronization/ISynchronizer';

const debug = Debug('@signageos/front-display:Applet:Sync:appletSyncSagas');

export const GroupStatusAction = 'Sync.GroupStatus';
type GroupStatusAction = GroupStatus & {
	type: typeof GroupStatusAction;
};

export const BroadcastedValueAction = 'Sync.BroadcastedValue';
type BroadcastedValueAction = BroadcastedValue & {
	type: typeof BroadcastedValueAction;
};

export const ClosedAction = 'Sync.Closed';
type ClosedAction = {
	type: typeof ClosedAction;
	error?: string;
};

export function* syncEventEmitting(window: Window, messageTypePrefix: string, getState: () => IFrontState, synchronizer: ISynchronizer) {
	const syncValueChannel = createChannel((putEvent: (message: BroadcastedValueAction | GroupStatusAction | ClosedAction) => void) => {
		synchronizer.addListener(SynchronizerEvent.GroupStatus, (event: GroupStatus) => {
			debug('got group status event', event);
			const action: GroupStatusAction = {
				type: GroupStatusAction,
				...event,
			};
			putEvent(action);
		});

		synchronizer.addListener(SynchronizerEvent.BroadcastedValue, (event: BroadcastedValue) => {
			debug('got broadcasted value event', event);
			const action: BroadcastedValueAction = {
				type: BroadcastedValueAction,
				...event,
			};
			putEvent(action);
		});

		synchronizer.addListener(SynchronizerEvent.Closed, (error?: Error) => {
			debug('got closed event', error);
			const action: ClosedAction = {
				type: ClosedAction,
				error: error?.message ?? undefined,
			};
			putEvent(action);
		});
	});

	// spawn process that will take sync events and create messages for applet from them
	yield spawn(takeEveryChannelMessage, syncValueChannel, function* (action: BroadcastedValueAction | GroupStatusAction | ClosedAction) {
		try {
			switch (action.type) {
				case GroupStatusAction:
					debug('sending group status event to applet', action);
					yield sendMessageToActiveAppletIfExists(window, getState, {
						type: messageTypePrefix + '.sync.device_status',
						connectedPeers: action.connectedPeers,
						groupName: action.groupName,
						isMaster: action.isMaster,
					});
					break;
				case BroadcastedValueAction:
					debug('sending broadcasted value event to applet', action);
					yield sendMessageToActiveAppletIfExists(window, getState, {
						type: messageTypePrefix + '.sync.set_value',
						groupName: action.groupName,
						key: action.key,
						value: action.value,
					});
					break;
				case ClosedAction:
					debug('sending closed event to applet', action);
					yield sendMessageToActiveAppletIfExists(window, getState, {
						type: messageTypePrefix + '.sync.closed',
						error: action.error,
					});
					break;
				default:
			}
		} catch (error) {
			console.error('sync set value event sending failed', error);
		}
	});
}
