import { call } from 'redux-saga/effects';
import HashAlgorithm from '../../../NativeDevice/HashAlgorithm';
import ICacheDriver from '../../../NativeDevice/ICacheDriver';
import { IFile } from '../../../NativeDevice/fileSystem';
import OfflineCache from '../../../OfflineCache/OfflineCache';
import AppletNativeCacheError from '../Error/AppletNativeCacheError';
import AppletResourceError from '../Error/AppletResourceError';
import ErrorCodes from '../Error/ErrorCodes';
import ErrorSuggestions from '../Error/ErrorSuggestions';
import InternalNativeCacheError from '../Error/InternalNativeCacheError';
import InternalOfflineCacheError from '../Error/InternalOfflineCacheError';
import { HandlerResult, IHandlerParams } from '../IHandler';
import IDecompressFileMessage from './IDecompressFileMessage';
import IDeleteContentMessage from './IDeleteContentMessage';
import IDeleteFileMessage from './IDeleteFileMessage';
import IGetChecksumMessage from './IGetChecksumMessage';
import ILoadContentMessage from './ILoadContentMessage';
import ILoadFileMessage from './ILoadFileMessage';
import ISaveContentMessage from './ISaveContentMessage';
import ISaveFileMessage from './ISaveFileMessage';
import IValidateChecksumMessage from './IValidateChecksumMessage';

function getAppletPrefix(appletUid: string, shortAppletFilesUrl: boolean) {
	return `applet/${shortAppletFilesUrl ? appletUid.slice(0, 8) : appletUid}/data/`;
}

export function* handleOfflineCacheMessage(
	messageTypePrefix: string,
	data:
		| ISaveFileMessage
		| ISaveContentMessage
		| ILoadFileMessage
		| ILoadContentMessage
		| IDeleteFileMessage
		| IDeleteContentMessage
		| IValidateChecksumMessage
		| IGetChecksumMessage
		| IDecompressFileMessage,
	nativeDriver: ICacheDriver,
	offlineCache: OfflineCache,
	appletPrefix: string,
	_timingChecksum: string,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.offline.cache.list_files':
			return yield call(handleFileSystemListFiles, offlineCache, appletPrefix);
		case messageTypePrefix + '.offline.cache.list_contents':
			return yield call(handleFileSystemListContents, nativeDriver, appletPrefix);
		case messageTypePrefix + '.offline.cache.save_file':
			return yield call(handleFileSystemSaveFile, data as ISaveFileMessage, offlineCache, appletPrefix);
		case messageTypePrefix + '.offline.cache.save_content':
			return yield call(handleFileSystemSaveContent, data as ISaveContentMessage, nativeDriver, appletPrefix);
		case messageTypePrefix + '.offline.cache.load_file':
			return yield call(handleFileSystemLoadFile, data as ILoadFileMessage, offlineCache, appletPrefix);
		case messageTypePrefix + '.offline.cache.load_content':
			return yield call(handleFileSystemLoadContent, data as ILoadContentMessage, nativeDriver, appletPrefix);
		case messageTypePrefix + '.offline.cache.delete_file':
			return yield call(handleFileSystemDeleteFile, data as IDeleteFileMessage, offlineCache, appletPrefix);
		case messageTypePrefix + '.offline.cache.delete_content':
			return yield call(handleFileSystemDeleteContent, data as IDeleteContentMessage, nativeDriver, appletPrefix);
		case messageTypePrefix + '.offline.cache.validate_checksum':
			return yield call(handleFileSystemValidateChecksum, data as IValidateChecksumMessage, offlineCache, appletPrefix);
		case messageTypePrefix + '.offline.cache.get_checksum':
			return yield call(handleFileSystemGetChecksum, data as IGetChecksumMessage, offlineCache, appletPrefix);
		case messageTypePrefix + '.offline.cache.decompress_file':
			return yield call(handleFileSystemDecompressFile, data as IDecompressFileMessage, offlineCache, appletPrefix);
		default:
			return null;
	}
}

async function handleFileSystemListFiles(offlineCache: OfflineCache, appletPrefix: string) {
	let fileUids: string[] = [];
	try {
		fileUids = await offlineCache.listFilesRecursively();
	} catch (error) {
		throw new InternalOfflineCacheError({
			kind: 'internalOfflineCacheErrorWithOrigin',
			message: "Couldn't not read the files from the offline cache.",
			code: ErrorCodes.OFFLINE_CACHE_LIST_FILES_ERROR,
			origin: 'listFilesRecursively',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	const appletFileUids: string[] = [];

	for (let fileUid of fileUids) {
		if (fileUid.indexOf(appletPrefix) === 0) {
			appletFileUids.push(fileUid.substring(appletPrefix.length));
		}
	}

	return {
		fileUids: appletFileUids,
	};
}

async function handleFileSystemListContents(nativeDriver: ICacheDriver, appletPrefix: string) {
	const appletContentUids: string[] = [];

	try {
		const contentUids = await nativeDriver.cacheGetUids();

		for (let contentUid of contentUids) {
			if (contentUid.indexOf(appletPrefix) === 0) {
				appletContentUids.push(contentUid.substring(appletPrefix.length));
			}
		}

		return {
			contentUids: appletContentUids,
		};
	} catch (error) {
		throw new InternalNativeCacheError({
			kind: 'internalNativeCacheErrorWithOrigin',
			message: 'Unexpected error ocurred when listing content',
			code: ErrorCodes.NATIVE_CACHE_GET_CONTENT_ERROR,
			origin: 'cacheGetUids',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

// muttable part
let savingFiles: { [uid: string]: ISaveFileMessage } = {};
async function handleFileSystemSaveFile(data: ISaveFileMessage, offlineCache: OfflineCache, appletPrefix: string) {
	const fileUid = appletPrefix + data.uid;
	if (typeof savingFiles[fileUid] !== 'undefined') {
		throw new AppletResourceError({
			kind: 'appletResourcesError',
			message: 'already loading file: ' + fileUid,
			code: ErrorCodes.FILE_IS_ALREADY_LOADING,
			suggestion: ErrorSuggestions.FILE_IS_ALREADY_LOADING,
		});
	}
	const exists = await offlineCache.fileExists(fileUid);
	if (exists) {
		throw new AppletResourceError({
			kind: 'appletResourcesError',
			message: 'already existing file: ' + fileUid,
			code: ErrorCodes.FILE_ALREADY_EXISTS,
			suggestion: ErrorSuggestions.FILE_ALREADY_EXISTS,
		});
	}

	savingFiles[fileUid] = data;
	console.info('start', fileUid);
	try {
		await offlineCache.retriableDownloadFile(5, fileUid, data.uri, data.headers);
	} finally {
		delete savingFiles[fileUid];
	}
	console.info('end', fileUid);

	const savedFile: IFile | null = await offlineCache.getFile(fileUid);

	if (savedFile === null) {
		throw new InternalOfflineCacheError({
			kind: 'internalOfflineCacheError',
			message: "File wasn't saved correctly.",
			code: ErrorCodes.OFFLINE_CACHE_SAVE_FILE_GET_FILE_ERROR,
		});
	}

	return {
		savedFile: { filePath: savedFile!.localUri },
	};
}

// muttable part
let savingContents: { [uid: string]: ISaveContentMessage } = {};
async function handleFileSystemSaveContent(data: ISaveContentMessage, nativeDriver: ICacheDriver, appletPrefix: string) {
	const contentUid = appletPrefix + data.uid;
	// TODO throw error if content already exists
	if (typeof savingContents[contentUid] !== 'undefined') {
		throw new AppletNativeCacheError({
			kind: 'appletNativeCacheError',
			message: 'Already saving the file with UID: ' + contentUid,
			code: ErrorCodes.NATIVE_CACHE_ALREADY_SAVING_CONTENT,
		});
	}

	savingContents[contentUid] = data;
	console.info('start', contentUid);
	try {
		await nativeDriver.cacheSave(contentUid, data.content);
	} catch (error) {
		throw new InternalNativeCacheError({
			kind: 'internalNativeCacheErrorWithOrigin',
			message: "Couldn't save the file to the offline cache.",
			code: ErrorCodes.NATIVE_CACHE_SAVE_CONTENT_ERROR,
			origin: `cacheSave(${contentUid}, ${data.content})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	} finally {
		delete savingContents[contentUid];
	}
	console.info('end', contentUid);
	return {
		savedContent: data,
	};
}

async function handleFileSystemLoadFile(data: ILoadFileMessage, offlineCache: OfflineCache, appletPrefix: string) {
	const fileUid = appletPrefix + data.uid;
	const loadedFile: IFile | null = await offlineCache.getFile(fileUid);

	if (loadedFile === null) {
		throw new InternalOfflineCacheError({
			kind: 'internalOfflineCacheError',
			message: 'Reading the file from the offline cache failed.',
			code: ErrorCodes.OFFLINE_CACHE_GET_FILE_ERROR,
		});
	}

	return {
		loadedFile: { filePath: loadedFile.localUri },
	};
}

async function handleFileSystemLoadContent(data: ILoadFileMessage, nativeDriver: ICacheDriver, appletPrefix: string) {
	const contentUid = appletPrefix + data.uid;
	try {
		return {
			loadedContent: await nativeDriver.cacheGet(contentUid),
		};
	} catch (error) {
		throw new InternalNativeCacheError({
			kind: 'internalNativeCacheErrorWithOrigin',
			message: "Couldn't load the file from offline cache.",
			code: ErrorCodes.NATIVE_CACHE_GET_CONTENT_ERROR,
			origin: `cacheGet(${contentUid})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function handleFileSystemDeleteFile(data: IDeleteFileMessage, offlineCache: OfflineCache, appletPrefix: string) {
	const fileUid = appletPrefix + data.uid;
	await offlineCache.deleteFileAndDeleteDirectoryIfEmpty(fileUid);
	return {};
}

async function handleFileSystemDeleteContent(data: IDeleteContentMessage, nativeDriver: ICacheDriver, appletPrefix: string) {
	const contentUid = appletPrefix + data.uid;
	try {
		await nativeDriver.cacheGet(contentUid); //  if the content doesn't exist, it will reject
	} catch (error) {
		throw new InternalNativeCacheError({
			kind: 'internalNativeCacheErrorWithOrigin',
			message: "Couldn't load the file before deleting it.",
			code: ErrorCodes.NATIVE_CACHE_GET_CONTENT_ERROR,
			origin: `cacheGet(${contentUid})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	try {
		await nativeDriver.cacheDelete(contentUid);
	} catch (error) {
		throw new InternalNativeCacheError({
			kind: 'internalNativeCacheErrorWithOrigin',
			message: "File wasn't deleted correctly.",
			code: ErrorCodes.NATIVE_CACHE_DELETE_CONTENT_ERROR,
			origin: `cacheDelete(${contentUid})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	return {};
}

async function handleFileSystemValidateChecksum(data: IValidateChecksumMessage, offlineCache: OfflineCache, appletPrefix: string) {
	const fileUid = appletPrefix + data.uid;
	const exists = await offlineCache.fileExists(fileUid);
	if (!exists) {
		throw new Error(`file ${fileUid} must be saved before hashing!`);
	}

	const hashAlgorithm = data.hashType;
	if (!isHashAlgorithmSupported(hashAlgorithm)) {
		throw new Error(`hash algorithm ${hashAlgorithm} is not supported`);
	}

	const checksum = await offlineCache.getFileChecksum(fileUid, hashAlgorithm as HashAlgorithm);
	return { validatedChecksum: checksum === data.hash };
}

async function handleFileSystemGetChecksum(data: IGetChecksumMessage, offlineCache: OfflineCache, appletPrefix: string) {
	const fileUid = appletPrefix + data.uid;
	const exists = await offlineCache.fileExists(fileUid);
	if (!exists) {
		throw new Error(`file ${fileUid} must be saved before hashing!`);
	}

	const hashAlgorithm = data.hashType;
	if (!isHashAlgorithmSupported(hashAlgorithm)) {
		throw new Error(`hash algorithm ${hashAlgorithm} is not supported`);
	}

	const checksum = await offlineCache.getFileChecksum(fileUid, hashAlgorithm as HashAlgorithm);
	return { checksum };
}

async function isHashAlgorithmSupported(hashAlgorithm: HashAlgorithm) {
	return Object.values(HashAlgorithm).includes(hashAlgorithm);
}

async function handleFileSystemDecompressFile(data: IDecompressFileMessage, offlineCache: OfflineCache, appletPrefix: string) {
	const fileUid = appletPrefix + data.uid;
	const fileDestinationUid = appletPrefix + data.destinationUid;
	const exists = await offlineCache.fileExists(fileUid);
	if (!exists) {
		throw new Error(`file ${fileUid} must be saved before decompressing!`);
	}
	await offlineCache.extractFile(fileUid, fileDestinationUid, data.method);

	return {};
}

export default function* offlineCacheHandler({
	messageTypePrefix,
	data,
	cacheDriver,
	offlineCache,
	appletUid,
	timingChecksum,
	shortAppletFilesUrl,
}: IHandlerParams): HandlerResult {
	const appletPrefix = getAppletPrefix(appletUid, shortAppletFilesUrl);
	return yield call(
		handleOfflineCacheMessage,
		messageTypePrefix,
		data as ISaveFileMessage | ISaveContentMessage | IValidateChecksumMessage | IGetChecksumMessage,
		cacheDriver,
		offlineCache,
		appletPrefix,
		timingChecksum,
	);
}
