import Orientation from '@signageos/common-types/dist/Device/Resolution/Orientation';
import VideoOrientation from '@signageos/common-types/dist/Device/Resolution/VideoOrientation';
import { NativeTimer } from '@signageos/common-types/dist/Device/Settings/DeviceSettings';
import { convertWeekday } from '@signageos/lib/dist/Timer/convertWeekday';
import { getObjectKeys } from '@signageos/lib/dist/Utils/object';
import Debug from 'debug';
import moment from 'moment-timezone';
import { awaitDependencies } from '../../../DI/dependencyInjection';
import { getCurrentTime, setNtpOrManualTime } from '../../../Management/Device/DateTime/dateTimeFacade';
import IBatteryStatus from '../../../NativeDevice/Battery/IBatteryStatus';
import IFrontDriver from '../../../NativeDevice/Front/IFrontDriver';
import { IAutoRecoverySettings } from '../../../NativeDevice/IAutoRecovery';
import IBrightness from '../../../NativeDevice/IBrightness';
import { IPeerRecoverySettings } from '../../../NativeDevice/IPeerRecovery';
import IManagementDriver from '../../../NativeDevice/Management/IManagementDriver';
import ManagementCapability from '../../../NativeDevice/Management/ManagementCapability';
import NativeFromDriverTimer from '../../../NativeDevice/Timer/ITimer';
import NumberTimerType, { numberToNativeTimerType } from '../../../NativeDevice/Timer/TimerType';
import Property from '../../../Property/Property';
import createPropertyStorage, { IPropertyStorage } from '../../../Property/propertyStorage';
import { IProprietaryTimerStorage } from '../../../Timer/ITimerStorage';
import { EmptyObject } from '../../../Util/EmptyObject';
import processCallback from '../../../Util/processCallback';
import { putDevicePinChanged } from '../../Security/controlEnablingSagas';
import { DEFAULT_OPEN_PIN_CODE } from '../../Security/security';
import AppletSecurityError from '../Error/AppletSecurityError';
import ErrorCodes from '../Error/ErrorCodes';
import ErrorSuggestions from '../Error/ErrorSuggestions';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { IProprietaryTimer } from './ProprietaryTimer/IProprietaryTimer';
import {
	IAppUpgradeMessage,
	IDeleteProprietaryTimerMessage,
	IFactoryResetMessage,
	IFirmwareUpgradeMessage,
	IGetApplicationTypeMessage,
	IGetApplicationVersionMessage,
	IGetAutoRecoveryMessage,
	IGetBatteryStatusMessage,
	IGetBrandMessage,
	IGetCurrentTemperatureMessage,
	IGetCurrentTimeWithTimezoneMessage,
	IGetFirmwareTypeMessage,
	IGetFirmwareVersionMessage,
	IGetIsDisplayPowerOnMessage,
	IGetIsRemoteControlEnabledMessage,
	IGetModelMessage,
	IGetPeerRecoveryMessage,
	IGetPinCodeMessage,
	IGetProprietaryTimersMessage,
	IGetScreenBrightnessMessage,
	IGetSerialNumberMessage,
	IGetTimersMessage,
	IGetVolumeMessage,
	IInitializeMessage,
	IInstallPackageMessage,
	IIsHardwareAccelerationEnabledMessage,
	IManagementSupportsMessage,
	IPowerOffDisplayMessage,
	IPowerOnDisplayMessage,
	IRebootSystemMessage,
	RemoveScheduledRebootMessage,
	IResetSettingsMessage,
	IResizeScreenMessage,
	IRestartAppMessage,
	ISetAutoRecoveryMessage,
	ISetDebugMessage,
	ISetExtendedManagementUrl,
	ISetHardwareAccelerationMessage,
	ISetManualTimeWithTimezoneMessage,
	ISetNTPTimeWithTimezoneMessage,
	ISetPeerRecoveryMessage,
	ISetPinCodeMessage,
	ISetProprietaryTimerMessage,
	ISetRemoteControlEnabledMessage,
	SetScheduledRebootActionMessage,
	ISetScreenBrightnessMessage,
	ISetTimerMessage,
	ISetVolumeMessage,
	IUploadScreenshotMessage,
} from './messageTypes';
import { ManagementCapabilities } from '@signageos/common-types/dist/Device/Capabilities/ManagementCapabilities';
import PowerActionType from '@signageos/actions/dist/Device/Power/PowerActionType';
import { ScheduledRebootActions } from '../../Device/Power/scheduledRules';

const debug = Debug('@signageos/front-display:Front:Applet:Management:appletManagementHandler');

const PREFIX = 'management';
const DEBUG_PREFIX = 'debug';

export function* handleManagementMessage(
	messageTypePrefix: string,
	data:
		| IManagementSupportsMessage
		| IInitializeMessage
		| IGetApplicationTypeMessage
		| IGetApplicationVersionMessage
		| IAppUpgradeMessage
		| IGetModelMessage
		| IGetBrandMessage
		| IGetSerialNumberMessage
		| IGetBatteryStatusMessage
		| IGetCurrentTemperatureMessage
		| IUploadScreenshotMessage
		| IFirmwareUpgradeMessage
		| IGetFirmwareVersionMessage
		| IGetFirmwareTypeMessage
		| IGetVolumeMessage
		| ISetVolumeMessage
		| ISetScreenBrightnessMessage
		| IGetScreenBrightnessMessage
		| IInstallPackageMessage
		| IRebootSystemMessage
		| IRestartAppMessage
		| IPowerOnDisplayMessage
		| IPowerOffDisplayMessage
		| IGetIsDisplayPowerOnMessage
		| IResizeScreenMessage
		| IGetTimersMessage
		| ISetTimerMessage
		| IGetProprietaryTimersMessage
		| ISetProprietaryTimerMessage
		| ISetRemoteControlEnabledMessage
		| IGetIsRemoteControlEnabledMessage
		| IGetCurrentTimeWithTimezoneMessage
		| ISetManualTimeWithTimezoneMessage
		| ISetNTPTimeWithTimezoneMessage
		| ISetDebugMessage
		| IResetSettingsMessage
		| IFactoryResetMessage
		| ISetPinCodeMessage
		| IGetPinCodeMessage
		| IGetPeerRecoveryMessage
		| ISetPeerRecoveryMessage
		| IGetAutoRecoveryMessage
		| ISetAutoRecoveryMessage
		| ISetHardwareAccelerationMessage
		| IIsHardwareAccelerationEnabledMessage
		| SetScheduledRebootActionMessage
		| RemoveScheduledRebootMessage,
	managementDriver: IManagementDriver,
	frontDriver: IFrontDriver,
	propertyStorage: IPropertyStorage,
	applicationVersion: string,
	timerStorage: IProprietaryTimerStorage,
): HandlerResult {
	switch (data.type) {
		case `${messageTypePrefix}.${PREFIX}.supports`:
			return yield isCapabilitySupported(data as IManagementSupportsMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.initialize`:
			return yield handleInitialization(data as IInitializeMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_application_type`:
			return yield handleGetApplicationType(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.app_upgrade`: {
			return yield handleAppUpgrade(data as IAppUpgradeMessage, managementDriver);
		}
		case `${messageTypePrefix}.${PREFIX}.get_application_version`:
			return yield Promise.resolve({ applicationVersion });

		case `${messageTypePrefix}.${PREFIX}.get_model`:
			return yield handleGetModel(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_serial_number`:
			return yield handleGetSerialNumber(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_battery_status`:
			return yield handleGetBatteryStatus(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_current_temperature`:
			return yield handleGetCurrentTemperature(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_brand`:
			return yield handleGetBrand(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.upload_screenshot`:
			return yield handleUploadScreenshot(data as IUploadScreenshotMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.firmware_upgrade`: {
			return yield handleFirmwareUpgrade(data as IFirmwareUpgradeMessage, managementDriver);
		}
		case `${messageTypePrefix}.${PREFIX}.get_firmware_version`:
			return yield handleGetFirmwareVersion(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_firmware_type`:
			return yield handleGetFirmwareType(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_volume`:
			return yield handleGetVolume(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_volume`:
			return yield handleSetVolume(data as ISetVolumeMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_screen_brightness`:
			return yield handleSetScreenBrightness(data as ISetScreenBrightnessMessage, managementDriver, propertyStorage);

		case `${messageTypePrefix}.${PREFIX}.get_screen_brightness`:
			return yield handleGetScreenBrightness(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.install_package`:
			return yield handleInstallPackage(data as IInstallPackageMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.reboot_system`:
			return yield handleRebootSystem(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.restart_app`:
			return yield handleRestartApp(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.power_on_display`:
			return yield handlePowerOnDisplay(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.power_off_display`:
			return yield handlePowerOffDisplay(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_is_display_power_on`:
			return yield handleGetIsDisplayPowerOn(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.resize_screen`:
			return yield handleResizeScreen(data as IResizeScreenMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_screen_orientation`:
			return yield handleGetScreenOrientation(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_timers`:
			return yield handleGetTimers(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_timer`:
			return yield handleSetTimer(data as ISetTimerMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_proprietary_timers`:
			return yield handleGetProprietaryTimers(timerStorage);

		case `${messageTypePrefix}.${PREFIX}.set_proprietary_timer`:
			return yield handleSetProprietaryTimer(data as ISetProprietaryTimerMessage, timerStorage);

		case `${messageTypePrefix}.${PREFIX}.delete_proprietary_timer`:
			return yield handleDeleteProprietaryTimer(data as IDeleteProprietaryTimerMessage, timerStorage);

		case `${messageTypePrefix}.${PREFIX}.set_remote_control_enabled`:
			return yield handleSetRemoteControlEnabled(data as ISetRemoteControlEnabledMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_is_remote_control_enabled`:
			return yield handleGetIsRemoteControlEnable(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_current_time_with_timezone`:
			return yield handleGetCurrentTimeWithTimezone(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_current_time_with_timezone`:
			return yield handleSetManualTimeWithTimezone(data as ISetManualTimeWithTimezoneMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_ntp_time_with_timezone`:
			return yield handleSetNTPTimeWithTimezone(data as ISetNTPTimeWithTimezoneMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_debug`:
			return yield handleSetDebug(data as ISetDebugMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.is_enabled`:
			return yield handleIsDebugEnabled(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.reset_settings`:
			return yield handleResetSettings(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.factory_reset`:
			return yield handleFactoryReset(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_pin_code`:
			return yield* handleSetPinCode(frontDriver, data as ISetPinCodeMessage);

		case `${messageTypePrefix}.${PREFIX}.get_pin_code`:
			return yield handleGetPinCode(frontDriver);

		case `${messageTypePrefix}.${PREFIX}.get_peer_recovery`:
			return yield handleGetPeerRecovery(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_peer_recovery`:
			return yield handleSetPeerRecovery(data as ISetPeerRecoveryMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_auto_recovery`:
			return yield handleGetAutoRecovery(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_extendedmanagement_url`:
			return yield handleGetExtendedManagementUrl(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_extendedmanagement_url`:
			return yield handleSetExtendedManagementUrl(data as ISetExtendedManagementUrl, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_auto_recovery`:
			return yield handleSetAutoRecovery(data as ISetAutoRecoveryMessage, managementDriver);

		case `${messageTypePrefix}.${DEBUG_PREFIX}.is_remote_enabled`:
			return yield handleRemoteDebug(propertyStorage);

		case `${messageTypePrefix}.${PREFIX}.set_hardware_acceleration`:
			return yield handleSetHardwareAcceleration(data as ISetHardwareAccelerationMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.is_hardware_acceleration_enabled`:
			return yield handleIsHardwareAccelerationEnabled(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.set_scheduled_reboot`:
			return yield handleSetScheduledRebootAction(data as SetScheduledRebootActionMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.get_scheduled_reboots`:
			return yield handleGetScheduledRebootActions(managementDriver);

		case `${messageTypePrefix}.${PREFIX}.remove_scheduled_reboot`:
			return yield handleRemoveScheduledRebootAction(data as RemoveScheduledRebootMessage, managementDriver);

		case `${messageTypePrefix}.${PREFIX}.clear_scheduled_reboots`:
			return yield handleClearScheduledRebootActions(managementDriver);

		default:
			return null;
	}
}

async function isCapabilitySupported(
	data: IManagementSupportsMessage,
	managementDriver: IManagementDriver,
): Promise<{ supports: boolean }> {
	const capability = ManagementCapability[data.capability];
	if (typeof capability === 'undefined') {
		throw new Error('Invalid management capability');
	}

	const supports = await managementDriver.managementSupports(capability);
	return { supports };
}

async function handleInitialization(data: IInitializeMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.initialize(data.staticBaseUrl);
	return {};
}

async function handleGetApplicationType(managementDriver: IManagementDriver): Promise<{ applicationType: string }> {
	const applicationType = await managementDriver.getApplicationType();
	return { applicationType };
}

function* handleAppUpgrade(data: IAppUpgradeMessage, managementDriver: IManagementDriver) {
	const { baseUrl, version, directUri } = data;
	if (directUri && (baseUrl || version)) {
		throw new Error('Invalid arguments for upgrade fw');
	}
	let finalizeCallback: undefined | (() => Promise<void>) = undefined;
	const url = baseUrl || directUri;
	if (!url) {
		const { staticBaseUrl } = yield awaitDependencies(['staticBaseUrl']);
		finalizeCallback = yield managementDriver.appUpgrade(staticBaseUrl, version);
	} else {
		finalizeCallback = yield managementDriver.appUpgrade(url, version);
	}
	processCallback(finalizeCallback, { debugTag: 'appUpgrade()' });
	return {};
}

async function handleGetModel(managementDriver: IManagementDriver): Promise<{ model: string }> {
	const model = await managementDriver.systemInfo.getModel();
	return { model };
}

async function handleGetSerialNumber(managementDriver: IManagementDriver): Promise<{ serialNumber: string }> {
	const serialNumber = await managementDriver.systemInfo.getSerialNumber();
	return { serialNumber };
}

async function handleGetBrand(managementDriver: IManagementDriver): Promise<{ brand: string }> {
	const brand = await managementDriver.systemInfo.getBrand();
	return { brand };
}

async function handleGetBatteryStatus(managementDriver: IManagementDriver): Promise<{ batteryStatus: IBatteryStatus }> {
	const batteryStatus = await managementDriver.batteryGetStatus();
	return { batteryStatus };
}

async function handleGetCurrentTemperature(managementDriver: IManagementDriver): Promise<{ currentTemperature: number }> {
	const currentTemperature = await managementDriver.getCurrentTemperature();
	return { currentTemperature };
}

async function handleUploadScreenshot(
	data: IUploadScreenshotMessage,
	managementDriver: IManagementDriver,
): Promise<{ screenshotUrl: string; aHash?: string }> {
	const { uploadBaseUrl, computeHash } = data;
	const { url, imageInformations } = await managementDriver.screenshotUpload(uploadBaseUrl, computeHash);
	return { screenshotUrl: url, aHash: imageInformations?.aHash };
}

function* handleFirmwareUpgrade(data: IFirmwareUpgradeMessage, managementDriver: IManagementDriver) {
	const { baseUrl, version, directUri } = data;
	if (directUri && (baseUrl || version)) {
		throw new Error('Invalid arguments for upgrade fw');
	}
	let finalizeCallback: undefined | (() => Promise<void>) = undefined;
	const url = baseUrl || directUri;
	if (!url) {
		const { staticBaseUrl } = yield awaitDependencies(['staticBaseUrl']);
		finalizeCallback = yield managementDriver.firmwareUpgrade(staticBaseUrl, version);
	} else {
		finalizeCallback = yield managementDriver.firmwareUpgrade(url, version);
	}
	processCallback(finalizeCallback, { debugTag: 'firmwareUpgrade()' });
	return {};
}

async function handleGetFirmwareVersion(managementDriver: IManagementDriver): Promise<{ firmwareVersion: string }> {
	const firmwareVersion = await managementDriver.firmwareGetVersion();
	return { firmwareVersion };
}

async function handleGetFirmwareType(managementDriver: IManagementDriver): Promise<{ firmwareType: string | null }> {
	const firmwareType = await managementDriver.firmwareGetType();
	return { firmwareType };
}

async function handleGetVolume(managementDriver: IManagementDriver): Promise<{ volume: number }> {
	const volume = await managementDriver.getVolume();
	return { volume };
}

async function handleSetScreenBrightness(
	data: ISetScreenBrightnessMessage,
	managementDriver: IManagementDriver,
	propertyStorage: IPropertyStorage,
): Promise<EmptyObject> {
	const { timeFrom1, brightness1, timeFrom2, brightness2 } = data;
	await managementDriver.screenSetBrightness(timeFrom1, brightness1, timeFrom2, brightness2);
	await propertyStorage.setValue(Property.BRIGHTNESS_SETTINGS, {
		timeFrom1: timeFrom1,
		brightness1: brightness1,
		timeFrom2: timeFrom2,
		brightness2: brightness2,
	});
	return {};
}

async function handleInstallPackage(data: IInstallPackageMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	const { baseUrl, packageName, version, build } = data;
	await managementDriver.packageInstall(baseUrl, packageName, version, build);
	return {};
}

async function handleGetScreenBrightness(managementDriver: IManagementDriver): Promise<{ screenBrightness: IBrightness }> {
	const screenBrightness = await managementDriver.screenGetBrightness();
	return { screenBrightness };
}

async function handleSetVolume(data: ISetVolumeMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	const { volume } = data;
	await managementDriver.setVolume(volume);
	return {};
}

async function handleRebootSystem(managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.systemReboot();
	return {};
}

async function handleRestartApp(managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.appRestart();
	return {};
}

async function handlePowerOnDisplay(managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.displayPowerOn();
	return {};
}

async function handlePowerOffDisplay(managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.displayPowerOff();
	return {};
}

async function handleGetIsDisplayPowerOn(managementDriver: IManagementDriver): Promise<{ isDisplayPowerOn: boolean }> {
	const isDisplayPowerOn = await managementDriver.displayIsPowerOn();
	return { isDisplayPowerOn };
}

async function handleResizeScreen(data: IResizeScreenMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	const { baseUrl, currentVersion } = data;
	const orientation = Orientation[data.orientation];
	const resolution =
		data.resolution === 'FULL_HD'
			? { width: 1920, height: 1080 }
			: data.resolution === 'HD_READY'
				? { width: 1280, height: 720 }
				: undefined;
	let videoOrientation = undefined;

	if (typeof orientation === 'undefined') {
		throw new Error('Invalid  orientation');
	}
	if (typeof resolution === 'undefined') {
		throw new Error('Invalid resolution');
	}
	if (data.videoOrientation) {
		videoOrientation = VideoOrientation[data.videoOrientation];
		if (typeof videoOrientation === 'undefined') {
			throw new Error('Invalid video orientation');
		}
	}

	const finalizeCallback = await managementDriver.screenResize(baseUrl, orientation, resolution, currentVersion, videoOrientation);
	processCallback(finalizeCallback, { debugTag: 'screenResize()' });
	return {};
}

async function handleGetScreenOrientation(managementDriver: IManagementDriver): Promise<{ screenOrientation: keyof typeof Orientation }> {
	debug('handleGetScreenOrientation');
	const currentOrientation = await managementDriver.getCurrentOrientation();
	return { screenOrientation: Orientation[currentOrientation] as keyof typeof Orientation };
}

async function handleGetTimers(managementDriver: IManagementDriver): Promise<{ timers: NativeTimer[] }> {
	const timers = await managementDriver.getTimers();
	const appletTimers = timers.map((timer: NativeFromDriverTimer) => {
		return {
			type: numberToNativeTimerType(timer.type),
			timeOn: timer.timeOn,
			timeOff: timer.timeOff,
			weekdays: convertWeekday.fromTimerWeekday(timer.weekdays).toShort(),
			volume: timer.volume,
		};
	});
	return { timers: appletTimers };
}

async function handleSetTimer(data: ISetTimerMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	const { timeOn, timeOff, volume } = data;
	const timerType = NumberTimerType[data.timerType];
	const weekdays = convertWeekday.fromShort(data.weekdays).toTimerWeekday();

	if (typeof timerType === 'undefined') {
		throw new Error('Invalid timer type');
	}

	weekdays.forEach((weekday) => {
		if (typeof weekday === 'undefined') {
			throw new Error('Invalid timer week day');
		}
	});
	await managementDriver.setTimer(timerType, timeOn, timeOff, weekdays, volume);
	return {};
}

async function handleGetProprietaryTimers(timerStorage: IProprietaryTimerStorage): Promise<{ timers: IProprietaryTimer[] }> {
	const timersSettingsObject = await timerStorage.getShortTimers();
	const timerTypes = getObjectKeys(timersSettingsObject);
	const timers = timerTypes.map((timerType): IProprietaryTimer => {
		const timerSettings = timersSettingsObject[timerType];
		return {
			type: timerSettings.type,
			timeOn: timerSettings.timeOn,
			timeOff: timerSettings.timeOff,
			weekdays: timerSettings.weekdays,
		};
	});
	return { timers };
}

async function handleSetProprietaryTimer(data: ISetProprietaryTimerMessage, timerStorage: IProprietaryTimerStorage): Promise<EmptyObject> {
	const { timeOn, timeOff, timerType, weekdays, keepAppletRunning } = data;
	if (timeOn === null && timeOff === null) {
		// Backward compatibility for deleting timers
		return handleDeleteProprietaryTimer(data, timerStorage);
	}

	if (typeof timerType === 'undefined') {
		throw new Error('Invalid timer type');
	}

	weekdays.forEach((weekday) => {
		if (typeof weekday === 'undefined') {
			throw new Error('Invalid timer week day');
		}
	});

	await timerStorage.setShortTimer({
		type: timerType,
		timeOn,
		timeOff,
		weekdays,
		keepAppletRunning,
	});
	return {};
}

async function handleDeleteProprietaryTimer(
	data: IDeleteProprietaryTimerMessage,
	timerStorage: IProprietaryTimerStorage,
): Promise<EmptyObject> {
	const { timerType } = data;
	if (typeof timerType === 'undefined') {
		throw new Error('Invalid timer type');
	}
	await timerStorage.deleteShortTimer(timerType);
	return {};
}

async function handleSetRemoteControlEnabled(
	data: ISetRemoteControlEnabledMessage,
	managementDriver: IManagementDriver,
): Promise<EmptyObject> {
	const { enabled } = data;
	const finalizeCallback = await managementDriver.remoteControlSetEnabled(enabled);
	processCallback(finalizeCallback);
	return {};
}

async function handleGetIsRemoteControlEnable(managementDriver: IManagementDriver): Promise<{ isRemoteControlEnabled: boolean }> {
	const isRemoteControlEnabled = await managementDriver.remoteControlIsEnabled();
	return { isRemoteControlEnabled };
}

async function handleGetCurrentTimeWithTimezone(
	managementDriver: IManagementDriver,
): Promise<{ currentTimeWithTimezone: { currentDate: Date; timezone?: string; ntpServer?: string; ntpEnabled?: boolean } }> {
	const currentTimeWithTimezone = await getCurrentTime(managementDriver);
	return {
		currentTimeWithTimezone: {
			currentDate: new Date(currentTimeWithTimezone.timestamp),
			timezone: currentTimeWithTimezone.timezone || undefined,
			ntpServer: currentTimeWithTimezone.ntpServer,
			ntpEnabled: currentTimeWithTimezone.ntpEnabled,
		},
	};
}

async function handleSetManualTimeWithTimezone(
	data: ISetManualTimeWithTimezoneMessage,
	managementDriver: IManagementDriver,
): Promise<EmptyObject> {
	let timestampMs: number;

	if (data.currentDate instanceof Date) {
		// currentDate as Date is deprecated and this logic stays here simply for backwards compatibility
		timestampMs = moment(data.currentDate).valueOf();
	} else {
		timestampMs = moment
			.tz(
				{
					year: data.currentDate.year,
					month: data.currentDate.month - 1,
					day: data.currentDate.day,
					hour: data.currentDate.hour,
					minute: data.currentDate.minute,
					second: data.currentDate.second,
				},
				data.timezone,
			)
			.valueOf();
	}

	const finalizeCallback = await setNtpOrManualTime(managementDriver, timestampMs, data.timezone);
	processCallback(finalizeCallback, { debugTag: 'setNtpOrManualTime()' });
	return {};
}

async function handleSetNTPTimeWithTimezone(data: ISetNTPTimeWithTimezoneMessage, managementDriver: IManagementDriver) {
	const newCurrentMoment = moment();
	const finalizeCallback = await setNtpOrManualTime(managementDriver, newCurrentMoment.valueOf(), data.timezone, data.ntpServer);
	processCallback(finalizeCallback, { debugTag: 'setNtpOrManualTime()' });
	return {};
}

async function handleSetDebug(data: ISetDebugMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	const { enabled } = data;
	const finalizeCallback = await managementDriver.setDebug(enabled);
	processCallback(finalizeCallback, { debugTag: 'setDebug()' });
	return {};
}

async function handleIsDebugEnabled(managementDriver: IManagementDriver): Promise<{ isEnabled: boolean }> {
	const isEnabled = await managementDriver.isDebugEnabled();
	return { isEnabled };
}

async function handleResetSettings(managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.resetSettings();
	return {};
}

async function handleFactoryReset(managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.factoryReset();
	return {};
}

function* handleSetPinCode(frontDriver: IFrontDriver, { pinCode }: ISetPinCodeMessage) {
	if (pinCode === DEFAULT_OPEN_PIN_CODE) {
		throw new AppletSecurityError({
			kind: 'appletSecurityError',
			code: ErrorCodes.SECURITY_PIN_CODE_0000,
			message: 'The PIN code 0000 is not allowed.',
		});
	}
	yield frontDriver.security.setPin(pinCode);
	yield putDevicePinChanged(pinCode);
	return {};
}

async function handleGetPinCode(frontDriver: IFrontDriver): Promise<{ pinCode: string }> {
	const pinCode = await frontDriver.security.getPin();
	if (!pinCode) {
		throw new AppletSecurityError({
			kind: 'appletSecurityError',
			code: ErrorCodes.SECURITY_PIN_CODE_NOT_SET_YET,
			message: 'The PIN code was not set yet.',
			suggestion: ErrorSuggestions.SECURITY_PIN_CODE_NOT_SET_YET,
		});
	}
	return { pinCode };
}

async function handleGetPeerRecovery(managementDriver: IManagementDriver): Promise<IPeerRecoverySettings> {
	return managementDriver.getPeerRecovery();
}

async function handleSetPeerRecovery(data: ISetPeerRecoveryMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.setPeerRecovery(
		data.enabled
			? { enabled: true, urlLauncherAddress: data.urlLauncherAddress }
			: { enabled: false, autoEnableTimeoutMs: data.autoEnableTimeoutMs },
	);
	return {};
}

async function handleGetAutoRecovery(managementDriver: IManagementDriver): Promise<IAutoRecoverySettings> {
	return managementDriver.getAutoRecovery();
}

async function handleSetAutoRecovery(data: ISetAutoRecoveryMessage, managementDriver: IManagementDriver): Promise<EmptyObject> {
	await managementDriver.setAutoRecovery(
		data.enabled
			? { enabled: true, healthcheckIntervalMs: data.healthcheckIntervalMs }
			: { enabled: false, autoEnableTimeoutMs: data.autoEnableTimeoutMs },
	);
	return {};
}

async function handleRemoteDebug(propertyStorage: IPropertyStorage) {
	const isEnabled = await propertyStorage.getValueOrDefault(Property.DEBUG_WEINRE_ENABLED, false);
	return { isEnabled };
}

async function handleGetExtendedManagementUrl(managementDriver: IManagementDriver) {
	const url = await managementDriver.getExtendedManagementUrl();
	return { url };
}

async function handleSetExtendedManagementUrl(data: ISetExtendedManagementUrl, managementDriver: IManagementDriver) {
	await managementDriver.setExtendedManagementUrl(data.url);

	return {};
}

async function handleSetHardwareAcceleration(data: ISetHardwareAccelerationMessage, managementDriver: IManagementDriver) {
	const isSupported = await managementDriver.managementSupports(ManagementCapabilities.HARDWARE_ACCELERATION);
	if (!isSupported) {
		throw new Error('Hardware acceleration is not supported on this device.');
	}
	await managementDriver.setHardwareAcceleration(data.enabled);
	return {};
}

async function handleIsHardwareAccelerationEnabled(managementDriver: IManagementDriver): Promise<{ enabled: boolean }> {
	const enabled = await managementDriver.isHardwareAccelerationEnabled();
	return { enabled };
}

async function handleSetScheduledRebootAction(data: SetScheduledRebootActionMessage, managementDriver: IManagementDriver) {
	const { rule } = data;
	// We need to generate unique id for each rule because there is
	// no backend what will do it for us
	const id = Math.random().toString(36).substring(2);
	await managementDriver.powerExecutors.powerActionTimer.set(id, {
		powerType: PowerActionType.SYSTEM_REBOOT,
		weekdays: rule.weekdays,
		time: rule.time,
	});
	return {};
}

async function handleGetScheduledRebootActions(managementDriver: IManagementDriver) {
	const rules = await managementDriver.powerExecutors.powerActionTimer.rules();
	const actions: ScheduledRebootActions[] = Object.entries(rules).map(([id, value]) => ({
		id,
		rule: {
			weekdays: value.weekdays,
			time: value.time,
		},
	}));
	return { actions };
}

async function handleRemoveScheduledRebootAction(data: RemoveScheduledRebootMessage, managementDriver: IManagementDriver) {
	await managementDriver.powerExecutors.powerActionTimer.unset(data.id);
	return {};
}

async function handleClearScheduledRebootActions(managementDriver: IManagementDriver) {
	await managementDriver.powerExecutors.powerActionTimer.reset();
	return {};
}

export default function* managementHandler({
	messageTypePrefix,
	data,
	managementDriver,
	frontDriver,
	cacheDriver,
	applicationVersion,
	timerStorage,
}: IHandlerParams): HandlerResult {
	const propertyStorage = createPropertyStorage(() => cacheDriver);
	return yield handleManagementMessage(
		messageTypePrefix,
		data as
			| IManagementSupportsMessage
			| IGetApplicationTypeMessage
			| IAppUpgradeMessage
			| IGetModelMessage
			| IGetBrandMessage
			| IGetSerialNumberMessage
			| IGetBatteryStatusMessage
			| IGetCurrentTemperatureMessage
			| IUploadScreenshotMessage
			| IFirmwareUpgradeMessage
			| IGetFirmwareVersionMessage
			| IGetVolumeMessage
			| ISetVolumeMessage
			| ISetScreenBrightnessMessage
			| IGetScreenBrightnessMessage
			| IInstallPackageMessage
			| IRebootSystemMessage
			| IRestartAppMessage
			| IPowerOnDisplayMessage
			| IPowerOffDisplayMessage
			| IGetIsDisplayPowerOnMessage
			| IResizeScreenMessage
			| ISetTimerMessage
			| ISetProprietaryTimerMessage
			| ISetRemoteControlEnabledMessage
			| IGetIsRemoteControlEnabledMessage
			| IGetCurrentTimeWithTimezoneMessage
			| ISetManualTimeWithTimezoneMessage
			| ISetNTPTimeWithTimezoneMessage
			| ISetDebugMessage
			| IResetSettingsMessage
			| IGetPeerRecoveryMessage
			| ISetPeerRecoveryMessage
			| IGetAutoRecoveryMessage
			| ISetAutoRecoveryMessage
			| ISetHardwareAccelerationMessage
			| IIsHardwareAccelerationEnabledMessage
			| SetScheduledRebootActionMessage
			| RemoveScheduledRebootMessage,
		managementDriver,
		frontDriver,
		propertyStorage,
		applicationVersion,
		timerStorage,
	);
}
