import IManagementDriver from '../../../../NativeDevice/Management/IManagementDriver';
import IIsWifiClientEnabledMessage from './IIsWifiClientEnabledMessage';
import IIsWifiConnectedMessage from './IIsWifiConnectedMessage';
import IEnableWifiClientMessage from './IEnableWifiClientMessage';
import IIsWifiAPEnabledMessage from './IIsWifiAPEnabledMessage';
import IEnableWifiAPMessage from './IEnableWifiAPMessage';
import IDisableWifiMessage from './IDisableWifiMessage';
import IConnectWifiMessage from './IConnectWifiMessage';
import IDisconnectWifiMessage from './IDisconnectWifiMessage';
import IWifiGetCountryMessage from './IWifiGetCountryMessage';
import IWifiSetCountryMessage from './IWifiSetCountryMessage';
import IScanWifiDevicesMessage from './IScanWifiDevicesMessage';
import { HandlerResult, IHandlerParams } from '../../IHandler';

export function* handleWifiMessage(
	messageTypePrefix: string,
	data:
		| IIsWifiClientEnabledMessage
		| IIsWifiConnectedMessage
		| IEnableWifiClientMessage
		| IIsWifiAPEnabledMessage
		| IEnableWifiAPMessage
		| IDisableWifiMessage
		| IConnectWifiMessage
		| IDisconnectWifiMessage
		| IWifiGetCountryMessage
		| IWifiSetCountryMessage
		| IScanWifiDevicesMessage,
	nativeDriver: IManagementDriver,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.management.wifi.is_enabled':
			return yield isWifiClientEnabled(nativeDriver);
		case messageTypePrefix + '.management.wifi.get_connected_to':
			return yield getWifiConnectedTo(nativeDriver);
		case messageTypePrefix + '.management.wifi.enable':
			return yield enableWifiClient(nativeDriver);
		case messageTypePrefix + '.management.wifi.disable':
			return yield disableWifi(nativeDriver);
		case messageTypePrefix + '.management.wifi.is_ap_enabled':
			return yield isWifiAPEnabled(nativeDriver);
		case messageTypePrefix + '.management.wifi.enable_ap':
			return yield enableWifiAP(data as IEnableWifiAPMessage, nativeDriver);
		case messageTypePrefix + '.management.wifi.connect':
			return yield connectWifi(data as IConnectWifiMessage, nativeDriver);
		case messageTypePrefix + '.management.wifi.disconnect':
			return yield disconnectWifi(nativeDriver);
		case messageTypePrefix + '.management.wifi.get_country':
			return yield getCountry(nativeDriver);
		case messageTypePrefix + '.management.wifi.set_country':
			return yield setCountry(data as IWifiSetCountryMessage, nativeDriver);
		case messageTypePrefix + '.management.wifi.scan':
			return yield scanWifiDevices(nativeDriver);
		default:
			return null;
	}
}

async function isWifiClientEnabled(nativeDriver: IManagementDriver) {
	const isEnabled = await nativeDriver.wifi.isClientEnabled();
	return { isWifiEnabled: isEnabled };
}

async function getWifiConnectedTo(nativeDriver: IManagementDriver) {
	const ssid = await nativeDriver.wifi.getConnectedTo();
	return { ssid };
}

async function enableWifiClient(nativeDriver: IManagementDriver) {
	await nativeDriver.wifi.enableClient();
	return {};
}

async function disableWifi(nativeDriver: IManagementDriver) {
	await nativeDriver.wifi.disable();
	return {};
}

async function isWifiAPEnabled(nativeDriver: IManagementDriver) {
	const isEnabled = await nativeDriver.wifi.isAPEnabled();
	return { isWifiAPEnabled: isEnabled };
}

async function enableWifiAP(data: IEnableWifiAPMessage, nativeDriver: IManagementDriver) {
	await nativeDriver.wifi.enableAP(data.ssid, data.password);
	return {};
}

async function connectWifi(data: IConnectWifiMessage, nativeDriver: IManagementDriver) {
	const password = data.password;
	if (password && (password.length < 8 || password.length > 63)) {
		throw new Error('Password must be between 8 and 63 characters long');
	}
	await nativeDriver.wifi.connect(data.ssid, data.password, data.options);
	return {};
}

async function disconnectWifi(nativeDriver: IManagementDriver) {
	await nativeDriver.wifi.disconnect();
	return {};
}

async function getCountry(nativeDriver: IManagementDriver) {
	const countryCode = await nativeDriver.wifi.getCountry();
	return { countryCode };
}

async function setCountry(data: IWifiSetCountryMessage, nativeDriver: IManagementDriver) {
	await nativeDriver.wifi.setCountry(data.countryCode);
	return {};
}

async function scanWifiDevices(nativeDriver: IManagementDriver) {
	const devices = await nativeDriver.wifi.scan();
	return { devices };
}

export default function* wifiHandler({ messageTypePrefix, data, managementDriver }: IHandlerParams): HandlerResult {
	return yield handleWifiMessage(
		messageTypePrefix,
		data as
			| IIsWifiClientEnabledMessage
			| IIsWifiConnectedMessage
			| IEnableWifiClientMessage
			| IIsWifiAPEnabledMessage
			| IEnableWifiAPMessage
			| IDisableWifiMessage
			| IConnectWifiMessage
			| IDisconnectWifiMessage
			| IWifiGetCountryMessage
			| IWifiSetCountryMessage
			| IScanWifiDevicesMessage,
		managementDriver,
	);
}
