import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import InternalHardwareError from '../Error/InternalHardwareError';
import ErrorCodes from '../Error/ErrorCodes';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { sendMessageToActiveAppletIfExists } from '../sendAppletMessage';
import { IFrontState } from '../../frontReducers';
import IHardwareBarcodeScannerGetVersionMessage from './IHardwareBarcodeScannerGetVersionMessage';
import IHardwareBarcodeScannerStartMessage from './IHardwareBarcodeScannerStartMessage';
import IHardwareBarcodeScannerStopMessage from './IHardwareBarcodeScannerStopMessage';

export const DEFAULT_SCANNER_ID = 0;

interface IBarcodeScannerOptions {
	timeout: number;
	cancelPrevious: boolean;
}

export interface IScanners {
	[scannerId: number]: { status: BarcodeScannerStatus };
}

export enum BarcodeScannerStatus {
	IDLE = 0,
	STARTING = 1,
	USED = 2,
	STOPPING = 3,
}

function handleHardwareBarcodeScannerMessage(
	scanners: IScanners,
	messageTypePrefix: string,
	msg: IHardwareBarcodeScannerGetVersionMessage | IHardwareBarcodeScannerStartMessage | IHardwareBarcodeScannerStopMessage,
	nativeDriver: IDriver,
	window: Window,
	getState: () => IFrontState,
) {
	const getMessage = (name: string) => `${messageTypePrefix}.hardware.barcode_scanner.${name}`;
	switch (msg.type) {
		case getMessage('get_version'):
			return getVersion(nativeDriver);
		case getMessage('start'):
			const startMsg = msg as IHardwareBarcodeScannerStartMessage;
			return startScanning(scanners, nativeDriver, window, getState, getMessage, startMsg.scannerId, startMsg.options);
		case getMessage('stop'):
			const stopMsg = msg as IHardwareBarcodeScannerStopMessage;
			return stopScanning(scanners, nativeDriver, stopMsg.scannerId);
		default:
			return Promise.resolve(null);
	}
}

async function getVersion(nativeDriver: IDriver) {
	try {
		const version = await nativeDriver.hardware.barcodeScanner.getVersion();
		return { version };
	} catch (error) {
		throw new InternalHardwareError({
			kind: 'internalHardwareErrorWithOrigin',
			message: 'Failed to get version',
			code: ErrorCodes.HARDWARE_GET_BARCODE_SCANNER_VERSION_ERROR,
			origin: 'getVersion()',
			originStack: error.stack,
			originMessage: error.message,
		});
	}
}

async function startScanning(
	scanners: IScanners,
	nativeDriver: IDriver,
	window: Window,
	getState: () => IFrontState,
	getMessage: (name: string) => string,
	scannerId: number,
	options: IBarcodeScannerOptions,
) {
	if (!scanners[scannerId]) {
		throw new InternalHardwareError({
			kind: 'internalHardwareError',
			message: 'Selected barcode scanner is unknown',
			code: ErrorCodes.HARDWARE_START_OF_UNKNOWN_BARCODE_SCANNER,
		});
	} else if (scanners[scannerId].status === BarcodeScannerStatus.USED && options.cancelPrevious) {
		await stopScanning(scanners, nativeDriver, scannerId);
	} else if (scanners[scannerId].status !== BarcodeScannerStatus.IDLE) {
		throw new InternalHardwareError({
			kind: 'internalHardwareError',
			message: 'Selected barcode scanner is busy',
			code: ErrorCodes.HARDWARE_START_OF_BUSY_BARCODE_SCANNER,
		});
	}
	scanners[scannerId].status = BarcodeScannerStatus.STARTING;
	try {
		const onData = (data: string) => {
			sendMessageToActiveAppletIfExists(window, getState, {
				type: getMessage('data'),
				scannerId,
				data,
			});
		};
		const onError = (message: string) => {
			sendMessageToActiveAppletIfExists(window, getState, {
				type: getMessage('error'),
				scannerId,
				data: message,
			});
		};
		await nativeDriver.hardware.barcodeScanner.startScanning(scannerId, options.timeout, onData, onError);
	} catch (error) {
		scanners[scannerId].status = BarcodeScannerStatus.IDLE;
		throw new InternalHardwareError({
			kind: 'internalHardwareErrorWithOrigin',
			message: 'Failed to start scanning barcode',
			code: ErrorCodes.HARDWARE_START_OF_BARCODE_SCANNER_ERROR,
			origin: `startScanning(${JSON.stringify(options)})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	scanners[scannerId] = { status: BarcodeScannerStatus.USED };
	return {};
}

async function stopScanning(scanners: IScanners, nativeDriver: IDriver, scannerId: number) {
	if (scanners[scannerId].status !== BarcodeScannerStatus.USED) {
		throw new InternalHardwareError({
			kind: 'internalHardwareError',
			message: 'Barcode scanner to stop is already idle',
			code: ErrorCodes.HARDWARE_STOP_OF_IDLE_BARCODE_SCANNER,
		});
	}
	scanners[scannerId].status = BarcodeScannerStatus.STOPPING;
	try {
		await nativeDriver.hardware.barcodeScanner.stopScanning(scannerId);
	} catch (error) {
		throw new InternalHardwareError({
			kind: 'internalHardwareErrorWithOrigin',
			message: 'Failed to stop barcode scanner',
			code: ErrorCodes.HARDWARE_STOP_OF_BARCODE_SCANNER_ERROR,
			origin: `stopScanning(${scannerId})`,
			originStack: error.stack,
			originMessage: error.message,
		});
	}
	scanners[scannerId].status = BarcodeScannerStatus.IDLE;
	return {};
}

const defaultScanners: IScanners = { [DEFAULT_SCANNER_ID]: { status: BarcodeScannerStatus.IDLE } };

export default function* hardwareBarcodeScannerHandler(
	{
		messageTypePrefix,
		data,
		frontDriver,
		window,
		getState,
	}: Pick<IHandlerParams, 'messageTypePrefix' | 'data' | 'frontDriver' | 'window' | 'getState'>,
	mockScanners?: IScanners,
): HandlerResult {
	const scanners = mockScanners ?? defaultScanners;
	return yield handleHardwareBarcodeScannerMessage(scanners, messageTypePrefix, data, frontDriver, window, getState);
}
