import _ from 'lodash';
import { put } from 'redux-saga/effects';
import IBrowserOpenLinkMessage from './IBrowserOpenLinkMessage';
import IBrowserOpenMessage from './IBrowserOpenMessage';
import IDriver from '../../../NativeDevice/Front/IFrontDriver';
import { ActiveOpenBrowserLink } from '@signageos/actions/dist/Applet/activeAppletActions';
import AppletBrowserError from '../Error/AppletBrowserError';
import ErrorCodes from '../Error/ErrorCodes';
import InternalBrowserError from '../Error/InternalBrowserError';
import { HandlerResult, IHandlerParams } from '../IHandler';
import { IOpenLinkOptions } from '../../../NativeDevice/IBrowser';

export function* handleBrowserMessage(
	messageTypePrefix: string,
	data: IBrowserOpenLinkMessage | IBrowserOpenMessage,
	nativeDriver: IDriver,
	appletUid: string,
	timingChecksum: string,
): HandlerResult {
	switch (data.type) {
		case messageTypePrefix + '.browser.open_link':
		case messageTypePrefix + '.browser.open':
			if (typeof data.uri !== 'string') {
				throw new AppletBrowserError({
					kind: 'appletBrowserError',
					message: 'Property argument uri must be a string.',
					code: ErrorCodes.BROWSER_LINK_IS_NOT_STRING,
				});
			}
			yield put({
				type: ActiveOpenBrowserLink,
				appletUid,
				timingChecksum,
				uri: data.uri,
			} as ActiveOpenBrowserLink);
			try {
				const options: IOpenLinkOptions = _.omit(data, 'type', 'uri');
				if ('headlessMode' in data && typeof data.headlessMode !== 'undefined') {
					// New JS API: Populate default values depending on newly passed headlessMode.
					// Old JS API: Leave implementation without headlessMode up to the platform/application.
					// Headless mode: Prevent the user from leaving and keep cache around for later.
					// UI mode: Allow the user to leave and clear cache between sessions.
					options.clearData ??= !data.headlessMode;
					options.canUserClose ??= !data.headlessMode;
				}
				if (data.type === messageTypePrefix + '.browser.open_link') {
					yield nativeDriver.browserOpenLink(data.uri, options);
				} else {
					yield nativeDriver.browser.open(data.uri, options);
				}
			} catch (error) {
				throw new InternalBrowserError({
					kind: 'internalBrowserErrorWithOrigin',
					message: `Unexpected error occurred when opening ${data.uri}`,
					code: ErrorCodes.BROWSER_OPEN_LINK_ERROR,
					origin: 'browserOpenLink',
					originStack: error.stack,
					originMessage: error.message,
				});
			}
			return {};
		case messageTypePrefix + '.browser.close':
			yield nativeDriver.browser.close();
			return {};
		default:
			return null;
	}
}

export default function* browserHandler({
	messageTypePrefix,
	data,
	frontDriver,
	appletUid,
	timingChecksum,
}: IHandlerParams): HandlerResult {
	return yield handleBrowserMessage(messageTypePrefix, data as IBrowserOpenLinkMessage, frontDriver, appletUid, timingChecksum);
}
