import * as querystring from 'querystring';
import IndexedDBFileSystem from './IndexedDB/IndexedDBFileSystem';
import ProprietaryFileSystem from './ProprietaryFileSystem';

enum FileSystemType {
	/**
	 * Standard technology based on IndexedDB and Cache API.
	 */
	IndexedDB = 'indexed_db',
	/**
	 * Obsolete technology based on File API.
	 * Available only in Chromium based browsers.
	 * https://www.w3.org/TR/file-system-api/
	 */
	Webkit = 'webkit',
}

export function createBrowserFileSystem(window: Window, namespace: string, storageBaseUrl: string) {
	const queryParams = parseQueryParams(window.location.search);
	let fileSystemType = queryParams.fileSystem;

	const supportsIndexedDB = 'indexedDB' in window && 'caches' in window;
	if (!fileSystemType && !supportsIndexedDB) {
		console.warn('Running emulator on very old system. Using legacy webkit filesystem instead of indexedDB filesystem.');
		fileSystemType = FileSystemType.Webkit;
	}

	if (fileSystemType === FileSystemType.Webkit) {
		handleWebkitFileSystemWarning(window);
		return new ProprietaryFileSystem(window, window.PERSISTENT, Number.MAX_SAFE_INTEGER);
	} else {
		handleIndexedDBFileSystemWarning(window);
		return new IndexedDBFileSystem(window.indexedDB, window.caches, storageBaseUrl, namespace);
	}
}

function parseQueryParams(search: string) {
	return querystring.parse(search.slice(1));
}

function handleWebkitFileSystemWarning(window: Window) {
	const requestFileSystem = window.requestFileSystem || window.webkitRequestFileSystem;
	if (!requestFileSystem) {
		console.warn(
			'File System API is not supported by this browser.\n' +
				'https://developer.mozilla.org/en-US/docs/Web/API/FileSystem\n\n' +
				'Consider using IndexedDB file system (add query parameter `?fileSystem=indexed_db` into the URL).\n',
		);
	}
}

function handleIndexedDBFileSystemWarning(window: Window) {
	if (!window.indexedDB) {
		console.warn(
			'IndexedDB is not supported by this browser.\n' +
				'https://developer.mozilla.org/en-US/docs/Web/API/IndexedDB_API\n\n' +
				'Consider using File System API (add query parameter `?fileSystem=webkit` into the URL).\n',
		);
	}
	if (!window.caches) {
		if (window.isSecureContext) {
			console.warn(
				'Cache API is not supported by this browser.\n' +
					'https://developer.mozilla.org/en-US/docs/Web/API/Cache\n\n' +
					'Consider using File System API (add query parameter `?fileSystem=webkit` into the URL).\n',
			);
		} else {
			console.warn(
				'Cache API requires a secure context.\n' +
					'https://developer.mozilla.org/en-US/docs/Web/API/Cache\n' +
					'https://developer.mozilla.org/en-US/docs/Web/Security/Secure_Contexts\n\n' +
					`Current origin ${window.location.origin} is not secure. You have to navigate to a secure origin.\n` +
					`Try following addresses:\n`,
			);

			const sslUrl = new URL(window.location.href);
			if (sslUrl.protocol === 'http:') {
				sslUrl.protocol = 'https';
				console.warn(sslUrl.toString());
			}

			const localhostUrl = new URL(window.location.href);
			if (localhostUrl.hostname !== 'localhost') {
				localhostUrl.hostname = 'localhost';
				console.warn(localhostUrl.toString());
			}

			const localUrl = new URL(window.location.href);
			if (localUrl.hostname !== '127.0.0.1') {
				localUrl.hostname = '127.0.0.1';
				console.warn(localUrl.toString());
			}
		}
	}
}
