import path from 'path';
import IFrontFileSystem, { IArchiveInfo, ICopyFileOptions, IMoveFileOptions } from '../../NativeDevice/IFileSystem';
import { IFilePath, IFile, IStorageUnit, IHeaders } from '../../NativeDevice/fileSystem';
import HashAlgorithm from '../../NativeDevice/HashAlgorithm';
import { NotSupportedMethodError } from '../../NativeDevice/Error/basicErrors';
import Debug from 'debug';
import { waitForSuccessEventsOrFailEvents } from '../../Events/eventsHelper';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import {
	DirectoryIsNotEmptyError,
	DirectoryIsNotReadableError,
	DirectoryIsNotWritableError,
	FileAlreadyExistsError,
	FileDoesNotExistError,
	FileIsNotListableError,
	InvalidFileStateError,
	RootDirectoryIsNotDeletableError,
} from '../errors';
import { getNativeResponse } from '../../polyfills/fetch';
import { extractZipBlob, getZipExtractSize } from '../../Archive/extractor';
import { getChecksumOfBlob } from '../../Hash/checksum';
import { locked } from '@signageos/lib/es6/Lock/lockedDecorator';

/** The Cache API needs to have native instance of Response however it can be overwritten with fetch polyfill */
const Response = getNativeResponse();

const debug = Debug('@signageos/front-display:FileSystem:IndexedDBFileSystem');

const HARD_LIMIT_BYTES = 4 * 1024 * 1024 * 1024; // 4GB

type MemoryAddress = string & { __brand: 'MemoryLocation' };
function MemoryAddress(address?: string) {
	return (address ?? generateUniqueHash()) as MemoryAddress;
}

enum TransactionMode {
	ReadOnly = 'readonly',
	ReadWrite = 'readwrite',
}

enum Index {
	ParentFilePath = 'parentFilePath',
}

enum FileType {
	File = 'file',
	Directory = 'directory',
}

/**
 * The special file that represents the root directory.
 */
interface IRootDirectoryMetadata {
	/** Root directory with filePath="" has no parent */
	parentFilePath: null;
	fileType: FileType.Directory;
}

interface IBaseMetadata {
	parentFilePath: string;
}

interface IDirectoryMetadata extends IBaseMetadata {
	fileType: FileType.Directory;
}

interface IFileMetadata extends IBaseMetadata {
	fileType: FileType.File;
	address: MemoryAddress;
}

type IMetadata = IDirectoryMetadata | IFileMetadata | IRootDirectoryMetadata;

interface Transaction {
	tx: IDBTransaction;
	objectStore: IDBObjectStore;
	commit(): void;
	abort(): void;
}

const INTERNAL_TYPE = 'internal';
const MEMORY_TYPE_PREFIX = '_memory_';
const LOCAL_PATH_PREFIX = '/indexed_db/';
export const DATABASE_PREFIX = 'fileSystem';

export default class IndexedDBFileSystem implements IFrontFileSystem {
	private db: IDBDatabase;
	private databaseName: string;
	private cache: Cache;

	constructor(
		private indexedDB: IDBFactory,
		private caches: CacheStorage,
		private baseUrl: string,
		private namespace?: string,
	) {}

	public async initialize() {
		this.databaseName = `${DATABASE_PREFIX}${this.namespace ? '_' + this.namespace : ''}`;
		debug('initialize', this.databaseName, this.indexedDB);

		const cache = await this.caches.open(this.databaseName);
		this.cache = cache;

		const request = this.indexedDB.open(this.databaseName, 1);

		type IDBOpenEvent = IDBVersionChangeEvent & { target: IDBOpenDBRequest };
		const upgradeEvent = await waitForSuccessEventsOrFailEvents<'success' | 'upgradeneeded', 'error', IDBOpenEvent>(
			request,
			['success', 'upgradeneeded'],
			['error'],
		);
		if (upgradeEvent.type === 'upgradeneeded' && upgradeEvent.target) {
			debug('upgradeneeded', upgradeEvent);
			const upgradeRequest = upgradeEvent.target;
			this.db = upgradeRequest.result;
			await this.ensureInternalStorageUnit();
			if (upgradeRequest.transaction) {
				upgradeRequest.transaction.commit();
				await waitForSuccessEventsOrFailEvents<'complete', never, Event>(upgradeRequest.transaction, ['complete'], []);
			}
		} else {
			debug('success', upgradeEvent);
			this.db = request.result;
		}
	}

	public async destroy() {
		this.db.close();
	}

	public async listFiles(directoryPath: IFilePath): Promise<IFilePath[]> {
		if (!(await this.isDirectory(directoryPath))) {
			throw new FileIsNotListableError(directoryPath);
		}
		const { objectStore, commit } = this.startTransaction(
			directoryPath.storageUnit,
			{ mode: TransactionMode.ReadOnly },
			'listFiles',
			directoryPath.filePath,
		);
		const keys = await this.doRequest<string[]>(objectStore.index(Index.ParentFilePath).getAllKeys(directoryPath.filePath));
		commit();
		if (!keys) {
			throw new InvalidFileStateError(directoryPath);
		}
		const filePaths = keys.map((filePath) => ({ storageUnit: directoryPath.storageUnit, filePath }));
		return filePaths;
	}

	public async getFile(filePath: IFilePath): Promise<IFile | null> {
		try {
			const { blob } = await this.readFileAsBlob(filePath);
			const localUri = this.createLocalUriWithBaseOrigin(filePath);
			const file: IFile = {
				localUri,
				mimeType: blob.type,
				sizeBytes: blob.size,
			};
			return file;
		} catch (error) {
			return null;
		}
	}

	public async readFile(filePath: IFilePath): Promise<string> {
		const { blob } = await this.readFileAsBlob(filePath);
		const contents = await blob.text();
		return contents;
	}

	public async writeFile(filePath: IFilePath, contents: string) {
		const byteArray = Buffer.from(contents, 'utf8');
		const blob = new Blob([byteArray], { type: 'text/plain' });
		await this.writeBlobAsFile(filePath, blob);
	}

	public async appendFile(filePath: IFilePath, contents: string): Promise<void> {
		if (!(await this.exists(filePath))) {
			await this.writeFile(filePath, contents);
		} else {
			const parentDirectoryPath = this.getParentFilePath(filePath);
			if (!(await this.exists(parentDirectoryPath))) {
				throw new FileDoesNotExistError(parentDirectoryPath);
			}
			const byteArray = Buffer.from(contents, 'utf8');
			const appendBlob = new Blob([byteArray], { type: 'text/plain' });
			try {
				const { blob: existingBlob, fileMetadata } = await this.readFileAsBlob(filePath);
				const blob = new Blob([existingBlob, appendBlob], { type: existingBlob.type });

				const { commit, memoryObjectStore } = this.startTransaction(
					filePath.storageUnit,
					{ mode: TransactionMode.ReadWrite, memory: true },
					'appendFile',
					filePath.filePath,
				);
				await this.doRequest(memoryObjectStore.put(blob, fileMetadata.address));
				commit();
				await this.cache.put(this.createLocalUri(filePath), new Response(blob));
			} catch (error) {
				if (error instanceof DirectoryIsNotReadableError) {
					throw new DirectoryIsNotWritableError(filePath);
				} else {
					throw error;
				}
			}
		}
	}

	public async exists(filePath: IFilePath): Promise<boolean> {
		const { objectStore, commit } = this.startTransaction(
			filePath.storageUnit,
			{ mode: TransactionMode.ReadOnly },
			'exists',
			filePath.filePath,
		);
		const key = await this.doRequest<string>(objectStore.getKey(filePath.filePath));
		commit();
		return key !== undefined;
	}

	public async downloadFile(filePath: IFilePath, sourceUri: string, headers?: IHeaders): Promise<void> {
		const response = await fetch(sourceUri, { headers });
		if (!response.ok) {
			throw new Error(`Not OK response during downloadFile ${sourceUri}`);
		}
		const blob = await response.blob();
		await this.writeBlobAsFile(filePath, blob);
	}

	public async uploadFile(filePath: IFilePath, uri: string, formKey: string, headers?: { [key: string]: string }): Promise<string> {
		const { blob } = await this.readFileAsBlob(filePath);
		const formData = new FormData();
		formData.append(formKey, blob);
		const response = await fetch(uri, { method: 'POST', body: formData, headers });
		if (response.ok) {
			try {
				return await response.text();
			} catch (e) {
				// ignore if can't read the body (probably form data)
				return '';
			}
		}
		throw new Error('Error while reading file');
	}

	public async deleteFile(filePath: IFilePath, recursive: boolean): Promise<void> {
		if (filePath.filePath === '') {
			throw new RootDirectoryIsNotDeletableError(filePath);
		}
		if (!(await this.exists(filePath))) {
			throw new FileDoesNotExistError(filePath);
		}
		if (!recursive) {
			if ((await this.isDirectory(filePath)) && (await this.listFiles(filePath)).length > 0) {
				throw new DirectoryIsNotEmptyError(filePath);
			}
		}

		const deleteFileRecursive = async (deleteKey: string) => {
			const { objectStore, memoryObjectStore, commit, abort } = this.startTransaction(
				filePath.storageUnit,
				{ mode: TransactionMode.ReadWrite, memory: true },
				'deleteFile',
				filePath.filePath,
			);
			const fileMetadata = await this.doRequest<IMetadata>(objectStore.get(deleteKey));
			const deleteFilePath = { storageUnit: filePath.storageUnit, filePath: deleteKey };
			if (!fileMetadata) {
				abort();
				throw new FileDoesNotExistError(deleteFilePath);
			}
			if (fileMetadata.fileType === FileType.Directory) {
				const [keys] = await Promise.all([
					this.doRequest<string[]>(objectStore.index(Index.ParentFilePath).getAllKeys(deleteKey)),
					this.doRequest(objectStore.delete(deleteKey)),
				]);
				commit();
				if (keys) {
					await Promise.all(keys.map((key) => deleteFileRecursive(key)));
				}
			} else {
				await Promise.all([this.doRequest(memoryObjectStore.delete(fileMetadata.address)), this.doRequest(objectStore.delete(deleteKey))]);
				commit();
			}

			await this.cache.delete(this.createLocalUri(deleteFilePath), { ignoreSearch: true, ignoreMethod: true });
		};

		await deleteFileRecursive(filePath.filePath);
	}

	public async copyFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options: ICopyFileOptions = {}): Promise<void> {
		if (!options.overwrite && (await this.exists(destinationFilePath))) {
			throw new FileAlreadyExistsError(destinationFilePath);
		}
		const destinationParentFilePath = this.getParentFilePath(destinationFilePath);
		if (!(await this.exists(destinationParentFilePath))) {
			throw new FileDoesNotExistError(destinationParentFilePath);
		}

		if (await this.isDirectory(sourceFilePath)) {
			await this.ensureDirectoryRecursive(destinationFilePath);
			const sourceSubFilePaths = await this.listFiles(sourceFilePath);
			await Promise.all(
				sourceSubFilePaths.map(async (sourceSubFilePath) => {
					const destSubFilePath = {
						storageUnit: destinationFilePath.storageUnit,
						filePath: destinationFilePath.filePath + '/' + path.basename(sourceSubFilePath.filePath),
					};
					await this.copyFile(sourceSubFilePath, destSubFilePath, options);
				}),
			);
		} else {
			const { blob } = await this.readFileAsBlob(sourceFilePath);
			if ((await this.exists(destinationFilePath)) && (await this.isDirectory(destinationFilePath))) {
				await this.deleteFile(destinationFilePath, true);
			}
			await this.writeBlobAsFile(destinationFilePath, blob);
		}
	}

	public async moveFile(sourceFilePath: IFilePath, destinationFilePath: IFilePath, options: IMoveFileOptions = {}): Promise<void> {
		await this.copyFile(sourceFilePath, destinationFilePath, options);
		await this.deleteFile(sourceFilePath, true);
	}

	public link(_sourceFilePath: IFilePath, _destinationFilePath: IFilePath): Promise<void> {
		throw new NotSupportedMethodError('link');
	}

	public async createDirectory(directoryPath: IFilePath): Promise<void> {
		const exists = await this.exists(directoryPath);
		if (exists) {
			throw new FileAlreadyExistsError(directoryPath);
		}
		const parentDirectoryPath = this.getParentFilePath(directoryPath);
		if (!(await this.exists(parentDirectoryPath))) {
			throw new FileDoesNotExistError(parentDirectoryPath);
		}
		if (!(await this.isDirectory(parentDirectoryPath))) {
			throw new FileIsNotListableError(parentDirectoryPath);
		}
		const { objectStore, commit } = this.startTransaction(
			directoryPath.storageUnit,
			{ mode: TransactionMode.ReadWrite },
			'createDirectory',
			directoryPath.filePath,
		);
		const fileMetadata: IDirectoryMetadata = {
			parentFilePath: this.getParentFilePath(directoryPath).filePath,
			fileType: FileType.Directory,
		};
		await this.doRequest(objectStore.put(fileMetadata, directoryPath.filePath));
		commit();
	}

	public async isDirectory(filePath: IFilePath): Promise<boolean> {
		const { objectStore, commit } = this.startTransaction(
			filePath.storageUnit,
			{ mode: TransactionMode.ReadOnly },
			'isDirectory',
			filePath.filePath,
		);
		const fileMetadata = await this.doRequest<IMetadata>(objectStore.get(filePath.filePath));
		commit();
		if (!fileMetadata) {
			throw new FileDoesNotExistError(filePath);
		}
		return fileMetadata.fileType === FileType.Directory;
	}

	public async extractFile(archiveFilePath: IFilePath, destinationDirectoryPath: IFilePath, method: string): Promise<void> {
		if (method !== 'zip') {
			throw new Error(`Unsupported archive method ${method}`);
		}
		const { blob } = await this.readFileAsBlob(archiveFilePath);
		if (!(await this.exists(destinationDirectoryPath))) {
			await this.createDirectory(destinationDirectoryPath);
		}
		await extractZipBlob(
			blob,
			(relativeFilePath: string) =>
				this.ensureDirectoryRecursive({
					storageUnit: destinationDirectoryPath.storageUnit,
					filePath: destinationDirectoryPath.filePath + '/' + relativeFilePath,
				}),
			(relativeFilePath: string, fileBlob: Blob) =>
				this.writeBlobAsFile(
					{
						storageUnit: destinationDirectoryPath.storageUnit,
						filePath: destinationDirectoryPath.filePath + '/' + relativeFilePath,
					},
					fileBlob,
				),
		);
	}

	public async createArchive(_archiveFilePath: IFilePath, _archiveEntries: IFilePath[]): Promise<void> {
		throw new NotSupportedMethodError('createArchive');
	}

	public async getFileChecksum(filePath: IFilePath, hashType: HashAlgorithm): Promise<string> {
		const { blob } = await this.readFileAsBlob(filePath);
		return await getChecksumOfBlob(blob, hashType);
	}

	public async listStorageUnits(): Promise<IStorageUnit[]> {
		const storageUnits: IStorageUnit[] = [];
		for (let i = 0; i < this.db.objectStoreNames.length; i++) {
			const type = this.db.objectStoreNames[i];
			if (!type.startsWith(MEMORY_TYPE_PREFIX)) {
				storageUnits.push({
					type,
					removable: type !== INTERNAL_TYPE,
					// TODO compute sizes based on real data
					capacity: HARD_LIMIT_BYTES,
					freeSpace: HARD_LIMIT_BYTES,
					usableSpace: HARD_LIMIT_BYTES,
				});
			}
		}
		debug('listStorageUnits', storageUnits);
		return storageUnits;
	}

	public onStorageUnitsChanged(_listener: () => void) {
		// Storages never change. Only one available
	}

	public async getArchiveInfo(archiveFilePath: IFilePath): Promise<IArchiveInfo> {
		const { blob } = await this.readFileAsBlob(archiveFilePath);
		const uncompressedSize = await getZipExtractSize(blob);
		return { uncompressedSize };
	}

	public async readFileAsBlob(filePath: IFilePath) {
		const { objectStore, memoryObjectStore, commit, abort } = this.startTransaction(
			filePath.storageUnit,
			{ mode: TransactionMode.ReadOnly, memory: true },
			'readFileAsBlob',
			filePath.filePath,
		);
		const fileMetadata = await this.doRequest<IMetadata>(objectStore.get(filePath.filePath));
		if (!fileMetadata) {
			abort();
			throw new FileDoesNotExistError(filePath);
		}
		if (fileMetadata.fileType === FileType.Directory) {
			abort();
			throw new DirectoryIsNotReadableError(filePath);
		}
		const blob = await this.doRequest<Blob>(memoryObjectStore.get(fileMetadata.address));
		if (!blob) {
			abort();
			throw new InvalidFileStateError(filePath);
		}
		commit();
		return { blob, fileMetadata };
	}

	public async writeBlobAsFile(filePath: IFilePath, blob: Blob) {
		const exists = await this.exists(filePath);
		if (exists && (await this.isDirectory(filePath))) {
			throw new DirectoryIsNotWritableError(filePath);
		}
		const parentDirectoryPath = this.getParentFilePath(filePath);
		if (!(await this.exists(parentDirectoryPath))) {
			throw new FileDoesNotExistError(parentDirectoryPath);
		}
		const { objectStore, memoryObjectStore, commit } = this.startTransaction(
			filePath.storageUnit,
			{ mode: TransactionMode.ReadWrite, memory: true },
			'writeBlobAsFile',
			filePath.filePath,
		);

		if (exists) {
			const existingFileMetadata = await this.doRequest<IFileMetadata>(objectStore.get(filePath.filePath));
			if (existingFileMetadata) {
				await this.doRequest(memoryObjectStore.delete(existingFileMetadata.address));
			}
		}

		const address = MemoryAddress();
		const fileMetadata: IFileMetadata = {
			parentFilePath: this.getParentFilePath(filePath).filePath,
			fileType: FileType.File,
			address,
		};
		await Promise.all([
			this.doRequest(memoryObjectStore.put(blob, address)),
			this.doRequest(objectStore.put(fileMetadata, filePath.filePath)),
		]);
		commit();
		await this.cache.put(this.createLocalUri(filePath), new Response(blob));
	}

	public async wipeout(): Promise<void> {
		const clearCachePromise = new Promise<void>(async (resolve: () => void) => {
			for (let key of await this.caches.keys()) {
				await this.caches.delete(key);
			}
			resolve();
		});
		this.destroy();
		await Promise.all([this.doRequest(this.indexedDB.deleteDatabase(this.databaseName)), clearCachePromise]);
	}

	@locked(function (this: IndexedDBFileSystem, directoryPath: IFilePath) {
		return this.createLocalUri(directoryPath);
	})
	private async ensureDirectoryRecursive(directoryPath: IFilePath) {
		if (!(await this.exists(directoryPath))) {
			await this.ensureDirectoryRecursive(this.getParentFilePath(directoryPath));
			await this.createDirectory(directoryPath);
		}
	}

	private async doRequest<T>(request: IDBRequest): Promise<T | undefined> {
		const event = await waitForSuccessEventsOrFailEvents<'success', 'error', Event & { target: IDBRequest }>(
			request,
			['success'],
			['error'],
		);
		return event.target.result;
	}

	private startTransaction<T extends boolean>(
		storageUnit: IStorageUnit,
		options: { memory?: T; mode: TransactionMode },
		...messages: unknown[]
	): T extends true ? { memoryObjectStore: IDBObjectStore } & Transaction : Transaction {
		const storeNames = [storageUnit.type];
		if (options.memory) {
			storeNames.push(this.getMemoryType(storageUnit.type));
		}
		debug('transaction start', ...messages);
		const tx = this.db.transaction(storeNames, options.mode);
		tx.addEventListener('complete', (event) => debug('transaction complete', ...messages, event.target));

		const objectStore = tx.objectStore(storageUnit.type);
		let memoryObjectStore: IDBObjectStore | undefined;
		if (options.memory) {
			const memoryStorageType = this.getMemoryType(storageUnit.type);
			memoryObjectStore = tx.objectStore(memoryStorageType);
		}

		return {
			tx,
			objectStore,
			memoryObjectStore,
			commit() {
				try {
					tx.commit();
					debug('transaction commit succeeded', ...messages);
				} catch (error) {
					debug('transaction commit failed', ...messages, error);
					throw error;
				}
			},
			abort() {
				try {
					tx.abort();
					debug('transaction abort succeeded', ...messages);
				} catch (error) {
					debug('transaction abort failed', ...messages, error);
					throw error;
				}
			},
		} as ReturnType<typeof this.startTransaction<T>>;
	}

	private async ensureInternalStorageUnit() {
		if (!this.db.objectStoreNames.contains(INTERNAL_TYPE)) {
			const objectStore = this.db.createObjectStore(INTERNAL_TYPE);
			objectStore.createIndex(Index.ParentFilePath, 'parentFilePath');
			const fileMetadata: IRootDirectoryMetadata = {
				parentFilePath: null,
				fileType: FileType.Directory,
			};
			await this.doRequest(objectStore.put(fileMetadata, ''));
		}
		const memoryInternalType = this.getMemoryType(INTERNAL_TYPE);
		if (!this.db.objectStoreNames.contains(memoryInternalType)) {
			this.db.createObjectStore(memoryInternalType);
		}
	}

	private getMemoryType(type: string) {
		return `${MEMORY_TYPE_PREFIX}${type}`;
	}

	private getParentFilePath(filePath: IFilePath): IFilePath {
		let parentPath = path.dirname(filePath.filePath);
		if (parentPath === '.') {
			parentPath = '';
		}
		return {
			storageUnit: filePath.storageUnit,
			filePath: parentPath,
		};
	}

	private createLocalUri(filePath: IFilePath) {
		const namespace = this.namespace ? this.namespace + '/' : '';
		return `${LOCAL_PATH_PREFIX}${namespace}${filePath.storageUnit.type}/${filePath.filePath}`;
	}

	private createLocalUriWithBaseOrigin(filePath: IFilePath) {
		const namespace = this.namespace ? this.namespace + '/' : '';
		const baseUrl = this.baseUrl.endsWith('/') ? this.baseUrl.substring(0, this.baseUrl.length - 1) : this.baseUrl;
		return `${baseUrl}${LOCAL_PATH_PREFIX}${namespace}${filePath.storageUnit.type}/${filePath.filePath}`;
	}
}
