import Responsibility from './Responsibility';

/**
 * Defines a responsibility policy.
 */
export interface IResponsibilities {
	/**
	 * Whether a responsibility is had according to this policy.
	 * @param responsibility Responsibility to check.
	 */
	has(responsibility: Responsibility): boolean;
}

/**
 * Create a responsibility policy having specified responsibilities.
 * @param responsibilities Present responsibilities.
 */
export function createPresentResponsibilities(...responsibilities: Responsibility[]): IResponsibilities {
	return {
		has: (responsibility: Responsibility) => {
			return responsibilities.includes(responsibility);
		},
	};
}

/**
 * Create a responsibility policy having *all other* than specified responsibilities.
 * @param responsibilities Absent responsibilities.
 */
export function createAbsentResponsibilities(...responsibilities: Responsibility[]): IResponsibilities {
	return {
		has: (responsibility: Responsibility) => {
			return !responsibilities.includes(responsibility);
		},
	};
}

/**
 * Create a responsibility policy having *all* responsibilities. This may be useful for testing.
 */
export function createAllResponsibilities(): IResponsibilities {
	return {
		has: () => {
			return true;
		},
	};
}

/**
 * Create a responsibility policy having *no* responsibilities. This may be useful for testing.
 */
export function createNoResponsibilities(): IResponsibilities {
	return {
		has: () => {
			return false;
		},
	};
}
