interface BrowserEventEmitter<TSuccessEvent extends string, TFailedEvent extends string> {
	addEventListener(type: TSuccessEvent | TFailedEvent, listener: (...args: unknown[]) => unknown): void;
	removeEventListener(type: TSuccessEvent | TFailedEvent, listener: (...args: unknown[]) => unknown): void;
}

/**
 * The same logic as waitForSuccessOrFailEvents of @signageos/lib/dist/Events/eventsHelper.ts
 * however the interface of browser event emitter is different from nodejs event emitter
 */
export function waitForSuccessOrFailEvents<TSuccessEvent extends string, TFailedEvent extends string, TResult, TError>(
	eventEmitter: BrowserEventEmitter<TSuccessEvent, TFailedEvent>,
	successEvent: TSuccessEvent,
	failEvent: TFailedEvent,
) {
	return waitForSuccessEventsOrFailEvents<TSuccessEvent, TFailedEvent, TResult, TError>(eventEmitter, [successEvent], [failEvent]);
}

/**
 * The same logic as waitForSuccessEventsOrFailEvents of @signageos/lib/dist/Events/eventsHelper.ts
 * however the interface of browser event emitter is different from nodejs event emitter
 */
export function waitForSuccessEventsOrFailEvents<TSuccessEvent extends string, TFailedEvent extends string, TResult, TError = unknown>(
	eventEmitter: BrowserEventEmitter<TSuccessEvent, TFailedEvent>,
	successEvents: TSuccessEvent[],
	failEvents: TFailedEvent[],
) {
	return new Promise<TResult>((resolve: (result: TResult) => void, reject: (error: TError) => void) => {
		let successListener: (result: TResult) => void;
		let failListener: (error: TError) => void;
		successListener = (result: TResult) => {
			failEvents.forEach((failedEventType: TFailedEvent) => eventEmitter.removeEventListener(failedEventType, failListener));
			resolve(result);
		};
		failListener = (error: TError) => {
			successEvents.forEach((successEventType: TSuccessEvent) => eventEmitter.removeEventListener(successEventType, successListener));
			reject(error);
		};
		successEvents.forEach((successEventType: TSuccessEvent) => eventEmitter.addEventListener(successEventType, successListener));
		failEvents.forEach((failedEventType: TFailedEvent) => eventEmitter.addEventListener(failedEventType, failListener));
	});
}
