import { isEqualWith, pick } from 'lodash';
import { Saga, SagaIterator, Task } from 'redux-saga';
import { take, fork, put, cancel } from 'redux-saga/effects';
import { Container, DynamicContainer } from './Container';
import { RequestDIContainer, UpdateDIContainer, UpdateDIDependency } from './dependenciesActions';

/**
 * Use this function any time the dynamic dependencies are changed.
 */
export function* injectDependencies(dependencies: Partial<DynamicContainer>) {
	yield put<UpdateDIDependency>({
		type: UpdateDIDependency,
		dependencies,
	});
}

/**
 * withDependencies will not call the wrapped saga. If you'd like to process or yield saga you have to call it.
 * E.g.:
 *
 * ```ts
 * const mySaga = withDependencies(['dep1'], function* ({ dep1 }) {});
 * yield mySaga();
 * // or
 * yield call(mySaga);
 * ```
 *
 * The `();` at the end is necessary otherwise nothing is happening and saga is frozen.
 */
export const withDependencies = <D extends keyof Container, Args extends unknown[]>(
	dependencyKeys: D[],
	saga: Saga<[Pick<Container, D>, ...Args]>,
): Saga<Args> =>
	function* (...args: Args): SagaIterator {
		let runningTask: Task | undefined;
		let lastContainer: Partial<Container> | undefined;
		let action: UpdateDIContainer;
		yield put<RequestDIContainer>({
			type: RequestDIContainer,
		});
		while ((action = yield take(UpdateDIContainer))) {
			const newContainer = pick(action.container, dependencyKeys);
			if (!isShallowEqual(lastContainer, newContainer) && hasAllRequiredDependencies<D>(dependencyKeys, newContainer)) {
				lastContainer = newContainer;
				if (runningTask) {
					yield cancel(runningTask);
				}
				runningTask = yield fork(saga, newContainer, ...args);
			}
		}
	};

/**
 * If you want to get dependencies as soon as they will be available.
 */
export function* awaitDependencies<D extends keyof Container>(dependencyKeys: D[]): SagaIterator {
	let action: UpdateDIContainer;
	yield put<RequestDIContainer>({
		type: RequestDIContainer,
	});
	while ((action = yield take(UpdateDIContainer))) {
		if (hasAllRequiredDependencies<D>(dependencyKeys, action.container)) {
			return pick(action.container, dependencyKeys);
		}
	}
}

function hasAllRequiredDependencies<D extends keyof Container>(
	dependencyKeys: D[],
	container: Partial<Container>,
): container is Pick<Container, D> {
	return dependencyKeys.every((key) => Object.keys(container).includes(key));
}

function isShallowEqual(a: Record<string, unknown> | undefined, b: Record<string, unknown> | undefined) {
	return isEqualWith(a, b, (aVal, bBal, index) => (index === undefined ? undefined : Object.is(aVal, bBal)));
}
