import { call, cancel, fork, takeEvery } from 'redux-saga/effects';
import { withDependencies } from '../../DI/dependencyInjection';
import {
	CustomScript,
	CustomScriptResult,
	isArchiveCustomScript,
	isInlineCustomScript,
} from '@signageos/common-types/dist/CustomScript/CustomScript';
import { checksumString } from '../../Hash/checksum';
import { IProprietaryTimerStorage } from '../../Timer/ITimerStorage';
import { buffers, Channel, channel, SagaIterator, Task } from 'redux-saga';
import IMessage from '../../Front/Applet/IMessage';
import IManagementDriver, { IFrontManagementDriver, IPlatformSpecificAPIs } from '../../NativeDevice/Management/IManagementDriver';
import sendAppletMessage from '../../Front/Applet/sendAppletMessage';
import Debug from 'debug';
import { createErrorTransferObject } from '../../Front/Applet/Error/errorHelper';
import { IFile } from '../../NativeDevice/fileSystem';
import { handleAppletManagementMessage } from '../../Front/Applet/appletMessageHandler';
import { HandlerResult } from '../../Front/Applet/IHandler';
import ICacheDriver from '../../NativeDevice/ICacheDriver';
import { generateUniqueHash } from '@signageos/lib/dist/Hash/generator';
import BrowserRuntimeHelper from './BrowserRuntimeHelper';

const debug = Debug('@signageos/front-display:customScriptsSaga');

export class BrowserRuntime {
	constructor() {
		this.runScriptInBrowser = this.runScriptInBrowser.bind(this);
		this.bindCustomScriptsMessages = this.bindCustomScriptsMessages.bind(this);
		this.handleCustomScriptsMessage = this.handleCustomScriptsMessage.bind(this);
	}

	*runScriptInBrowser(
		managementDriver: IManagementDriver & IFrontManagementDriver,
		customScript: CustomScript,
		configuration: Record<string, any>,
		scriptFile?: IFile,
		appletJsFile?: IFile,
	) {
		const iframeId: string | undefined = getIframeUID(customScript);
		if (!iframeId) {
			throw new Error("Iframe's id cannot be generated");
		}
		const messageChannel = channel<MessageEvent>(buffers.dropping(10));
		const messagesHandler = (event: MessageEvent) => {
			messageChannel.put(event);
		};
		const bindCustomScriptsMessagesTask: Task = yield fork(this.bindCustomScriptsMessages, messageChannel, window, iframeId);

		try {
			window.addEventListener('message', messagesHandler);

			const result: CustomScriptResult = yield call(
				this.execute,
				customScript,
				configuration,
				iframeId,
				managementDriver.platformSpecificAPIs,
				scriptFile,
				appletJsFile,
			);
			debug('script executed OK: ' + result);
			return result;
		} catch (error) {
			throw error;
		} finally {
			yield cancel(bindCustomScriptsMessagesTask);
			window.removeEventListener('message', messagesHandler);
		}
	}

	public async execute(
		script: CustomScript,
		config: Record<string, any>,
		iframeId: string = generateUniqueHash(),
		platformSpecificAPIs: IPlatformSpecificAPIs,
		customScriptFile?: IFile,
		appletJsFile?: IFile,
	): Promise<CustomScriptResult> {
		if (isInlineCustomScript(script)) {
			debug('ProprietaryScripts.execute: ', script.script);
		} else {
			debug('ProprietaryScripts.execute: ', customScriptFile?.localUri);
		}
		const postResultID = generateUniqueHash();
		const postResult = (messageResult: string) => parent.postMessage({ result: messageResult, iframeId, postResultID }, '*');

		let iframe: HTMLIFrameElement;
		if (isInlineCustomScript(script)) {
			iframe = BrowserRuntimeHelper.createIframe(
				`${appletJsFile ? BrowserRuntimeHelper.wrapScript(appletJsFile.localUri) + BrowserRuntimeHelper.addConfigToSos() : ''}<script>${script.script}</script>`,
			);
		} else {
			if (!customScriptFile) {
				throw new Error('Custom script file is not provided');
			}
			let wrappedScript: string = '';
			if (appletJsFile) {
				wrappedScript += BrowserRuntimeHelper.wrapScript(appletJsFile.localUri);
			}
			wrappedScript += BrowserRuntimeHelper.addConfigToSos();
			wrappedScript += BrowserRuntimeHelper.wrapScript(customScriptFile.localUri);
			iframe = BrowserRuntimeHelper.createIframe(wrappedScript);
		}

		iframe.id = iframeId;
		const { messageListener, getResult } = BrowserRuntimeHelper.createMessageHandler(iframeId, postResultID);

		try {
			window.addEventListener('message', messageListener);
			BrowserRuntimeHelper.appendIframe(iframe);
			// contentWindow is available after appending iframe.
			BrowserRuntimeHelper.checkIframeWindow(iframe);

			BrowserRuntimeHelper.injectPlatformSpecificAPIs(iframe, postResult, config, platformSpecificAPIs);

			return await getResult();
		} catch (error) {
			return BrowserRuntimeHelper.parseError(script, error);
		} finally {
			BrowserRuntimeHelper.performCleanUp(messageListener, iframe);
		}
	}

	*bindCustomScriptsMessages(messageChannel: Channel<MessageEvent>, window: Window, iframeId: string) {
		const handleCustomScriptsMessage = this.handleCustomScriptsMessage;

		const callback = function* (
			managementDriver: IManagementDriver,
			cacheDriver: ICacheDriver,
			applicationVersion: string,
			timeStorage: IProprietaryTimerStorage,
			event: MessageEvent<any>,
		): SagaIterator {
			try {
				const data: IMessage = event.data;
				const iframeElement = window.document.getElementById(iframeId) as HTMLIFrameElement | null;
				if (!iframeElement) {
					return;
				}
				const invocationUid = data.invocationUid;
				try {
					const response = yield call(
						handleCustomScriptsMessage,
						window,
						data,
						managementDriver,
						cacheDriver,
						applicationVersion,
						timeStorage,
					);
					debug('message OK: ' + data.type, data);
					yield call(sendAppletMessage, window, iframeId, { type: 'hug.invocation.success', invocationUid, ...response });
				} catch (error) {
					yield call(sendAppletMessage, window, iframeId, {
						type: 'hug.invocation.error',
						invocationUid,
						error: createErrorTransferObject(error),
					});
					debug('message FAILED: ' + data.type, data, error);
				}
			} catch (error) {
				debug('bindCustomScriptsMessages failed', error);
			}
		};

		yield fork(
			withDependencies(
				['managementDriver', 'cacheDriver', 'applicationVersion', 'proprietaryTimerStorage'],
				function* ({ managementDriver, cacheDriver, applicationVersion, proprietaryTimerStorage }) {
					yield takeEvery(messageChannel, callback, managementDriver, cacheDriver, applicationVersion, proprietaryTimerStorage);
				},
			),
		);
	}

	*handleCustomScriptsMessage(
		window: Window,
		data: IMessage,
		managementDriver: IManagementDriver,
		cacheDriver: ICacheDriver,
		applicationVersion: string,
		timerStorage: IProprietaryTimerStorage,
	): HandlerResult {
		debug('Handling custom script message', data);
		return yield handleAppletManagementMessage({
			data,
			window,
			messageTypePrefix: 'hug',
			managementDriver,
			cacheDriver,
			applicationVersion,
			timerStorage,
		});
	}
}

function getIframeUID(customScript: CustomScript): string | undefined {
	if (isArchiveCustomScript(customScript)) {
		return customScript.md5Checksum;
	} else {
		return checksumString(customScript.script + Date.now());
	}
}
