import { IFile } from '../NativeDevice/fileSystem';
import OfflineCache from '../OfflineCache/OfflineCache';

export class CustomScriptDownloader {
	constructor(
		private offlineCache: OfflineCache,
		private getStaticBaseUrl: () => string,
	) {}

	public async getOrDownloadCustomScript(archiveUri: string, mainFileName: string, md5Checksum: string): Promise<IFile | null> {
		const customScriptLocalFolder = this.getCustomScriptDirectory(archiveUri, md5Checksum);

		const existsCustomScriptFile = await this.offlineCache.fileExists(customScriptLocalFolder);

		if (!existsCustomScriptFile) {
			await this.downloadCustomScriptToLocalStorage(archiveUri, customScriptLocalFolder, mainFileName);
		}

		return await this.offlineCache.getFile(customScriptLocalFolder + '/' + mainFileName);
	}

	private async downloadCustomScriptToLocalStorage(customScriptFilePath: string, localDirPath: string, mainFileName: string) {
		const relativePath = this.removeStaticUrl(customScriptFilePath);
		await this.ensureCustomScriptIsDownloaded(relativePath);
		await this.ensureCustomScriptIsExtracted(relativePath, localDirPath);

		const mainFilePath = localDirPath + '/' + mainFileName;
		return await this.offlineCache.getFile(mainFilePath);
	}

	private async ensureCustomScriptIsDownloaded(customScriptFileName: string) {
		const existsCustomScriptFolder = await this.offlineCache.fileExists(customScriptFileName);
		if (!existsCustomScriptFolder) {
			const staticBaseUrl = this.getStaticBaseUrl();
			const customScriptUri = staticBaseUrl + '/custom-script/' + customScriptFileName;
			await this.offlineCache.retriableDownloadFile(5, customScriptFileName, customScriptUri, undefined, true);
		}
	}

	private async ensureCustomScriptIsExtracted(customScriptFilePath: string, customScriptDirectoryPath: string) {
		const existsCustomScript = await this.offlineCache.fileExists(customScriptDirectoryPath);

		const customScriptFile = await this.offlineCache.getFile(customScriptFilePath);

		if (!existsCustomScript && customScriptFile) {
			await this.offlineCache.extractFile(customScriptFilePath, customScriptDirectoryPath, 'zip', true);
		}
	}

	private getCustomScriptDirectory(archiveUri: string, md5Checksum: string): string {
		const relativePath = this.removeStaticUrl(archiveUri);
		return relativePath.substring(0, relativePath.lastIndexOf('/')) + '/' + md5Checksum;
	}

	private removeStaticUrl(archiveUri: string): string {
		const matches = archiveUri.match(/custom-script\/(.+)$/);
		return matches ? matches[1] : archiveUri;
	}
}
