import * as URL from 'url';
import Debug from 'debug';
const debug = Debug('@signageos/front-display:Browser:IframeBrowser');

export type AclMode = 'blacklist' | 'whitelist';

export const createAclValidator =
	(iframe: HTMLIFrameElement, aclMode: AclMode, aclDomains: string[], denyCallback: () => void, allowCallback?: () => void) => () => {
		if (!iframe.contentWindow) {
			throw new Error(`New window is not available: ${iframe.src}`);
		}
		debug('load iframe url', iframe.contentWindow.location.href);
		if (!isUrlAllowed(iframe.contentWindow.location.href, aclMode, aclDomains)) {
			denyCallback();
		} else {
			allowCallback?.();
		}
	};

export function forbidUrlChangeByAcl(
	iframe: HTMLIFrameElement,
	defaultUri: string,
	aclMode: AclMode,
	aclDomains: string[],
	denyCallback: () => void,
) {
	// reinitiate unload listener, when page changed
	const allowCallback = () => forbidUrlChangeByAcl(iframe, defaultUri, aclMode, aclDomains, denyCallback);
	const aclValidator = createAclValidator(iframe, aclMode, aclDomains, denyCallback, allowCallback);
	if (!iframe.contentWindow) {
		throw new Error(`Current window is not available: ${iframe.src}`);
	}
	const currentContentWindow = iframe.contentWindow;
	const unloadListener = () => {
		// listen only once, because for next page it will be reinitiated
		currentContentWindow.removeEventListener('unload', unloadListener);
		debug('unload iframe url', currentContentWindow.location.href);
		setTimeout(aclValidator, 0); // wait a tick when iframe has new URL already
	};
	iframe.contentWindow.addEventListener('unload', unloadListener);
}

export function isUrlAllowed(url: string | undefined, aclMode: AclMode, aclDomains: string[]) {
	if (!url) {
		return false;
	}

	const urlMatchRules = aclDomains?.findIndex((aclDomain: string) => isUrlInDomain(url, aclDomain)) > -1;

	return aclMode === 'blacklist' ? !urlMatchRules : urlMatchRules;
}

export function isUrlInDomain(url: string, domain: string) {
	const parsedUrl = URL.parse(url);
	const urlDomain = parsedUrl.hostname; // excluding port
	return urlDomain?.endsWith(domain);
}
