import { EventEmitter } from 'events';
import { debug } from '@signageos/lib/dist/Debug/debugDecorator';
import { IHTMLElementProducer } from '../HTML/IHTMLElementProducer';
import IBrowser, { CloseReason, Event, EventType, IOpenLinkOptions } from '../NativeDevice/IBrowser';
import { forbidUrlChangeByAcl, createAclValidator, isUrlAllowed } from './iframeHelper';

const DEBUG_NAMESPACE = '@signageos/front-display:Browser:IframeBrowser';

const ACL_BACKUP_CHECK_INTERVAL_MS = 3e3;

enum InternalEvent {
	NewIframe = 'new_iframe',
}

/**
 * Implements Browser API via iframes
 */
export default class IframeBrowser implements IBrowser, IHTMLElementProducer {
	private iframe: HTMLIFrameElement | null = null;
	private aclInterval: any | null = null;
	private events: EventEmitter = new EventEmitter();
	private internalEvents: EventEmitter = new EventEmitter();

	constructor(private window: Window) {}

	@debug(DEBUG_NAMESPACE)
	public async open(uri: string, options?: IOpenLinkOptions): Promise<void> {
		if (options?.aclMode && !isUrlAllowed(uri, options.aclMode, options.aclDomains ?? [])) {
			throw new Error(`Default URL of browser ${uri} is not allowed.`);
		}
		if (this.iframe) {
			this.removeIframe();
		}

		let iframe: HTMLIFrameElement;
		if (options && options.coordinates) {
			const { x, y, width, height } = options.coordinates;
			iframe = this.createWindowedIframe(uri, x, y, width, height);
		} else {
			iframe = this.createFullscreenIframe(uri);
		}

		this.window.document.body.appendChild(iframe);
		this.iframe = iframe;
		this.internalEvents.emit(InternalEvent.NewIframe, this.iframe);

		if (options?.aclMode) {
			const denyCallback = () => this.open(uri, options);
			forbidUrlChangeByAcl(iframe, uri, options.aclMode, options.aclDomains ?? [], denyCallback);
			const aclValidator = createAclValidator(iframe, options.aclMode, options.aclDomains ?? [], denyCallback);
			this.aclInterval = setInterval(aclValidator, ACL_BACKUP_CHECK_INTERVAL_MS);
		}
	}

	@debug(DEBUG_NAMESPACE)
	public async close(): Promise<void> {
		if (this.iframe) {
			this.removeIframe();
			this.events.emit(EventType.CLOSE, {
				type: EventType.CLOSE,
				reason: CloseReason.API,
			});
		}
		if (this.aclInterval) {
			clearInterval(this.aclInterval);
			this.aclInterval = null;
		}
	}

	@debug(DEBUG_NAMESPACE)
	public addListener<TType extends EventType>(eventName: TType, listener: (event: Event<TType>) => void): void {
		this.events.addListener(eventName, listener);
	}

	@debug(DEBUG_NAMESPACE)
	public removeListener<TType extends EventType>(eventName: TType, listener: (event: Event<TType>) => void): void {
		this.events.removeListener(eventName, listener);
	}

	@debug(DEBUG_NAMESPACE)
	public onElementCreated(listener: (element: HTMLElement) => void): void {
		this.internalEvents.on(InternalEvent.NewIframe, listener);
	}

	public async isSupported(): Promise<boolean> {
		return true;
	}

	private createFullscreenIframe(uri: string) {
		const iframe = this.createBaseIframe(uri);
		iframe.style.width = this.window.document.body.clientWidth + 'px';
		iframe.style.height = this.window.document.body.clientHeight + 'px';
		iframe.style.border = '0';
		iframe.style.top = '0';
		iframe.style.left = '0';
		iframe.style.background = '#ffffff';
		return iframe;
	}

	private createWindowedIframe(uri: string, x: number, y: number, width: number, height: number) {
		const iframe = this.createBaseIframe(uri);
		iframe.style.width = width + 'px';
		iframe.style.height = height + 'px';
		iframe.style.border = '0';
		iframe.style.top = y + 'px';
		iframe.style.left = x + 'px';
		iframe.style.background = '#ffffff';
		return iframe;
	}

	private createBaseIframe(uri: string): HTMLIFrameElement {
		const iframe: HTMLIFrameElement = this.window.document.createElement('iframe');
		iframe.src = uri;
		iframe.style.width = '100vw';
		iframe.style.height = '100vh';
		iframe.style.position = 'absolute';
		iframe.style.border = '0';
		iframe.style.zIndex = '100';
		iframe.style.webkitTransform = 'translateZ(0)';
		iframe.style.transform = 'translateZ(0)';
		iframe.style.background = '#ffffff';
		try {
			iframe.sandbox.value = 'allow-forms allow-scripts allow-same-origin';
		} catch (error) {
			console.warn('Cannot set sandbox value of iframe', error);
		}
		return iframe;
	}

	private removeIframe() {
		if (!this.iframe) {
			throw new Error("iframe doesn't exist");
		}
		this.window.document.body.removeChild(this.iframe);
		this.iframe = null;
	}
}
